/*
 * Author : Christopher Henard (christopher.henard@uni.lu)
 * Date : 01/03/14
 * Copyright 2013 University of Luxembourg – Interdisciplinary Centre for Security Reliability and Trust (SnT)
 * All rights reserved
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package filtered;

import java.io.BufferedReader;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.sat4j.minisat.core.IOrder;
import org.sat4j.minisat.core.Solver;
import org.sat4j.minisat.orders.NegativeLiteralSelectionStrategy;
import org.sat4j.minisat.orders.PositiveLiteralSelectionStrategy;
import org.sat4j.minisat.orders.RandomLiteralSelectionStrategy;
import org.sat4j.minisat.orders.RandomWalkDecorator;
import org.sat4j.minisat.orders.VarOrderHeap;
import org.sat4j.reader.DimacsReader;
import org.sat4j.specs.ISolver;
import org.sat4j.tools.ModelIterator;
import org.sat4j.minisat.SolverFactory;

/**
 *
 * @author chris
 */
public class Filtered {

    private static double[] cost;
    private static boolean[] used_before;
    private static int[] defects;
    private static int numFeatures;
    private static int numConstraints;
    private static List<List<Integer>> constraints;
    private static Random r = new Random();
    private static final int SATtimeout = 1000;
    private static final long iteratorTimeout = 150000;

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        try {
            long start = System.currentTimeMillis();
            String fm = args[0];
            String augment = fm + ".augment";
            loadFM(fm, augment);

            int duration = Integer.parseInt(args[1]);

            Set<boolean[]> prods = new HashSet<boolean[]>();
            int evaluations = 0;

            while (System.currentTimeMillis() - start < duration) {

                boolean[] prod = randomProduct(fm);
                if (numViolatedConstraints(prod) == 0) // prods.add(randomProduct(fm));
                {
                    prods.add(prod);
                }
                evaluations ++;
            }

            long elapsed = System.currentTimeMillis() - start;

            System.out.println("RunTimeMS: " + elapsed);
            System.out.println("Evaluations: " + evaluations);

            int i = 0;

            for (boolean[] prod : prods) {
                printProd(i++, prod);
            }

            for (boolean[] prod : prods) {
                printProdEval(prod);
            }
        } catch (Exception ex) {
            System.out.println("Usage: java -jar filtered.jar fmDimacs timeMS");
        }

    }

    public static void printProd(int index, boolean[] prod) {
        System.out.print("Conf" + (index + 1) + ": ");
        for (int i = 0; i < prod.length; i++) {
            boolean b = prod[i];
            if (b) {
                System.out.print("1");
            } else {
                System.out.print("0");
            }

        }
        System.out.println("");
    }

    public static boolean[] randomProduct(String fm) {

        boolean[] prod = new boolean[numFeatures];
        //pure random products
        for (int i = 0; i < prod.length; i++) {
            prod[i] = r.nextBoolean();

        }

        int rand = r.nextInt(3);

        try {
            IOrder order;
            if (rand == 0) {
                order = new RandomWalkDecorator(new VarOrderHeap(new NegativeLiteralSelectionStrategy()), 1);
            } else if (rand == 1) {
                order = new RandomWalkDecorator(new VarOrderHeap(new PositiveLiteralSelectionStrategy()), 1);
            } else {
                order = new RandomWalkDecorator(new VarOrderHeap(new RandomLiteralSelectionStrategy()), 1);
            }

            ISolver dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
            dimacsSolver.setTimeout(SATtimeout);
            DimacsReader dr = new DimacsReader(dimacsSolver);
            dr.parseInstance(new FileReader(fm));

            ((Solver) dimacsSolver).setOrder(order);

            ISolver solverIterator = new ModelIterator(dimacsSolver);
            solverIterator.setTimeoutMs(iteratorTimeout);

            if (solverIterator.isSatisfiable()) {
                int[] i = solverIterator.model();

                for (int j = 0; j < i.length; j++) {
                    int feat = i[j];

                    int posFeat = feat > 0 ? feat : -feat;
//                    

                    prod[posFeat - 1] = feat > 0;

//                    else
//                    {
//                         prod[nFeat-1] = r.nextBoolean();
//                    }
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return prod;
    }

    public static int numViolatedConstraints(boolean[] b) {

        //IVecInt v = bitSetToVecInt(b);
        int s = 0;
        for (List<Integer> constraint : constraints) {

            boolean sat = false;

            for (Integer i : constraint) {
                int abs = (i < 0) ? -i : i;
                boolean sign = i > 0;
                if (b[abs - 1] == sign) {
                    sat = true;
                    break;
                }
            }
            if (!sat) {
                s++;
            }

        }

        return s;
    }

    public static void printProdEval(boolean[] prod) throws Exception {

        int unselected = 0, unused = 0, defect = 0;
        double cost_ = 0.0;

        for (int i = 0; i < prod.length; i++) {

            boolean b = prod[i];

            if (!b) {
                unselected++;
            } else {
                cost_ += cost[i];
                if (used_before[i]) {
                    defect += defects[i];
                } else {
                    unused++;
                }
            }

        }

        System.out.println(numViolatedConstraints(prod) + " " + unselected + " " + unused + " " + defect + " " + cost_);
    }

    public static void loadFM(String fm, String augment) throws Exception {

        BufferedReader in = new BufferedReader(new FileReader(fm));
        String line;
        while ((line = in.readLine()) != null) {
            line = line.trim();

            if (line.startsWith("p")) {
                StringTokenizer st = new StringTokenizer(line, " ");
                st.nextToken();
                st.nextToken();
                numFeatures = Integer.parseInt(st.nextToken());
                numConstraints = Integer.parseInt(st.nextToken());
                constraints = new ArrayList<List<Integer>>(numConstraints);
            }

            if (!line.startsWith("c") && !line.startsWith("p") && !line.isEmpty()) {
                StringTokenizer st = new StringTokenizer(line, " ");
                List<Integer> constraint = new ArrayList<Integer>(st.countTokens() - 1);

                while (st.hasMoreTokens()) {
                    int i = Integer.parseInt(st.nextToken());
                    if (i != 0) {
                        constraint.add(i);
                    }
                }
                constraints.add(constraint);
            }
        }
        in.close();

        cost = new double[numFeatures];
        used_before = new boolean[numFeatures];
        defects = new int[numFeatures];

        in = new BufferedReader(new FileReader(augment));
        while ((line = in.readLine()) != null) {
            line = line.trim();
            if (!line.startsWith("#")) {
                StringTokenizer st = new StringTokenizer(line, " ");
                int featIndex = Integer.parseInt(st.nextToken()) - 1;
                cost[featIndex] = Double.parseDouble(st.nextToken());
                used_before[featIndex] = Integer.parseInt(st.nextToken()) == 1;
                defects[featIndex] = Integer.parseInt(st.nextToken());
            }
        }
        in.close();

    }
}
