/*
 * Author : Christopher Henard (christopher.henard@uni.lu)
 * Date : 12/12/13
 * Copyright 2013 University of Luxembourg – Interdisciplinary Centre for Security Reliability and Trust (SnT)
 * All rights reserved
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package mutalog.core;

import java.util.ArrayList;
import java.util.List;

public class CNFFormula {

    protected List<Clause> clauses;
    protected int nVars;

    public CNFFormula() {
        clauses = new ArrayList<Clause>();
    }

    public CNFFormula(CNFFormula formula) {
        this();
        for (Clause c : formula.getClauses()) {
            clauses.add(new Clause(c));
        }
    }

    public int nClauses() {
        return clauses.size();
    }

    public int nVars() {
        return nVars;
    }

    public void setnVars(int nVars) {
        this.nVars = nVars;
    }

    public Clause getIClause(int index) {
        return clauses.get(index);
    }

    public void removeIClause(int index) {
        clauses.remove(index);
    }

    public void negtateIClause(int index) {
        Clause c = clauses.get(index);
        for (Literal l : c.getLiterals()) {
            Clause cl = new Clause();
            cl.addLiteral(l);
            cl.getILiteral(0).negate();
            clauses.add(cl);
        }
        clauses.remove(index);
    }

    public List<Clause> getClauses() {
        return clauses;
    }

    public void addClause(Clause c) {
        clauses.add(c);
    }

    public List<Mutant> getLiteralOmissionMutants() {

        List<Mutant> mutants = new ArrayList<Mutant>();

        int nClause = 0;
        for (Clause c : clauses) {
            int nLiteral = 0;
            for (Literal l : c.getLiterals()) {
                Mutant mutant = new Mutant(this, Mutant.LIT_OM);
                mutant.getIClause(nClause).removeLiteral(nLiteral);
                if (mutant.getIClause(nClause).isEmpty()) {
                    mutant.removeIClause(nClause);
                }
                nLiteral++;
                mutants.add(mutant);
            }


            nClause++;
        }
        return mutants;
    }

    public List<Mutant> getLiteralNegationMutants() {

        List<Mutant> mutants = new ArrayList<Mutant>();

        int nClause = 0;
        for (Clause c : clauses) {
            int nLiteral = 0;
            for (Literal l : c.getLiterals()) {
                Mutant mutant = new Mutant(this, Mutant.LIT_NEG);
                mutant.getIClause(nClause).getILiteral(nLiteral).negate();
                nLiteral++;
                mutants.add(mutant);
            }


            nClause++;
        }
        return mutants;
    }

    public List<Mutant> getClauseOmissionMutants() {

        List<Mutant> mutants = new ArrayList<Mutant>();

        int nClause = 0;
        for (Clause c : clauses) {

            Mutant mutant = new Mutant(this, Mutant.CLAUSE_OM);
            mutant.removeIClause(nClause);
            mutants.add(mutant);
            nClause++;
        }

        return mutants;
    }

    public List<Mutant> getClauseNegationMutants() {

        List<Mutant> mutants = new ArrayList<Mutant>();

        int nClause = 0;
        for (Clause c : clauses) {

            Mutant mutant = new Mutant(this, Mutant.CLAUSE_NEG);
            mutant.negtateIClause(nClause);
            mutants.add(mutant);
            nClause++;
        }

        return mutants;
    }

    public List<Mutant> getOperatorReferenceAndMutants() {

        List<Mutant> mutants = new ArrayList<Mutant>();

        int nClause = clauses.size();
        for (int i = 0; i < nClause - 1; i++) {

            Clause c = new Clause();
            for (Literal l : clauses.get(i).getLiterals()) {
                c.addLiteral(l);
            }
            for (Literal l : clauses.get(i + 1).getLiterals()) {
                c.addLiteral(l);
            }

            Mutant mutant = new Mutant(this, Mutant.OP_AND);
            mutant.removeIClause(i);
            mutant.removeIClause(i);
            mutant.addClause(c);
            mutants.add(mutant);
        }



        return mutants;
    }

    public List<Mutant> getOperatorReferenceOrMutants() {

        List<Mutant> mutants = new ArrayList<Mutant>();

        int nClause = 0;
        for (Clause c : clauses) {

            if (c.nLiterals() > 1) {
                int nLiteral = c.nLiterals();

                for (int i = 0; i < nLiteral - 1; i++) {
                    Mutant mutant = new Mutant(this, Mutant.OP_OR);
                    mutant.removeIClause(nClause);
                    Clause c1 = new Clause();
                    Clause c2 = new Clause();
                    for (int j = 0; j <= i; j++) {
                        c1.addLiteral(c.getILiteral(j));
                    }
                    for (int j = i + 1; j < nLiteral; j++) {
                        c2.addLiteral(c.getILiteral(j));

                    }

                    mutant.addClause(c1);
                    mutant.addClause(c2);

                    mutants.add(mutant);

                }
            }
            nClause++;
        }

        return mutants;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        int n = 0;
        for (int i = 0; i < clauses.size(); i++) {
            Clause c = clauses.get(i);
            if (n++ != 0) {
                sb.append("\n");
            }
            sb.append(c.toString());
        }
        return sb.toString();

    }
}
