/*
 * Author : Christopher Henard (christopher.henard@uni.lu)
 * Date : 12/12/13
 * Copyright 2013 University of Luxembourg – Interdisciplinary Centre for Security Reliability and Trust (SnT)
 * All rights reserved
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package mutalog.core;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Observable;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.sat4j.minisat.SolverFactory;
import org.sat4j.specs.ISolver;
import org.sat4j.specs.IVecInt;
import org.sat4j.specs.TimeoutException;

public class ModelMutaLog extends Observable {

    private static final String GLOBAL_ACTION_LOAD_FORMULA = "Loading the CNF formula";
    private static final String CURRENT_ACTION_LOAD_CLAUSES = "Parsing the clauses...";
    private static final String GLOBAL_ACTION_LOAD_TEST_SUITE = "Loading the test suite";
    private static final String CURRENT_ACTION_LOAD_TEST = "Parsing the tests...";
    private static final String GLOBAL_ACTION_CREATE_MUTANTS = "Creating mutants";
    private static final String CURRENT_ACTION_LITERAL_OMISSION_MUTANT = "Literal omission mutants...";
    private static final String CURRENT_ACTION_LITERAL_NEGATION_MUTANT = "Literal negation mutants...";
    private static final String CURRENT_ACTION_CLAUSE_OMISSION_MUTANT = "Clause omission mutants...";
    private static final String CURRENT_ACTION_CLAUSE_NEGATION_MUTANT = "Clause negation mutants...";
    private static final String CURRENT_ACTION_OPERATOR_AND_MUTANT = "Operator AND mutants...";
    private static final String CURRENT_ACTION_OPERATOR_OR_MUTANT = "Operator OR mutants...";
    private static final String GLOBAL_ACTION_EVAL_TEST_SUITE = "Evaluating the test suite";
    private static final String CURRENT_ACTION_EVAL_MUTANTS = "Evaluating the mutants...";
    private CNFFormula formula;
    private TestSuite testSuite;
    private List<Mutant> mutants;
    private String globalAction, currentAction;
    private boolean running, indeterminate;
    private int progress;
    private int killed, totalOk;

    public ModelMutaLog() {
        running = false;
        indeterminate = true;
        progress = 0;
        mutants = new ArrayList<Mutant>();
    }

    public boolean isRunning() {
        return running;
    }

    public void setRunning(boolean running) {
        this.running = running;
        if (!running) {
            indeterminate = true;
        }
        progress = 0;
        setChanged();
        notifyObservers();
    }

    public int getProgress() {
        return progress;
    }

    public void setProgress(int progress) {
        this.progress = progress;
        setChanged();
        notifyObservers();
    }

    public boolean isIndeterminate() {
        return indeterminate;
    }

    public void setIndeterminate(boolean indeterminate) {
        this.indeterminate = indeterminate;
        setChanged();
        notifyObservers();
    }

    public String getGlobalAction() {
        return globalAction;
    }

    public void setGlobalAction(String globalAction) {
        this.globalAction = globalAction;
        setChanged();
        notifyObservers();
    }

    public String getCurrentAction() {
        return currentAction;
    }

    public void setCurrentAction(String currentAction) {
        this.currentAction = currentAction;
        setChanged();
        notifyObservers();
    }

    public void parseCNFFormula(String file) {
        setRunning(true);
        setGlobalAction(GLOBAL_ACTION_LOAD_FORMULA);
        setCurrentAction(CURRENT_ACTION_LOAD_CLAUSES);
        formula = new CNFFormula();
        testSuite = null;
        mutants.clear();
        try {
            BufferedReader in = new BufferedReader(new FileReader(file));
            String line;
            while ((line = in.readLine()) != null) {
                line = line.trim();
                if (!line.startsWith("c")) {

                    StringTokenizer st = new StringTokenizer(line, " ");

                    if (line.startsWith("p")) {
                        st.nextToken();
                        st.nextToken();
                        formula.setnVars(Integer.parseInt(st.nextToken()));
                    } else {

                        int token;
                        Clause clause = new Clause(st.countTokens());
                        while (st.hasMoreTokens() && (token = Integer.parseInt(st.nextToken())) != 0) {
                            clause.addLiteral(new Literal(token));
                        }
                        if (!clause.isEmpty()) {
                            formula.addClause(clause);
                        }
                    }

                }
            }
            in.close();
        } catch (FileNotFoundException ex) {
            Logger.getLogger(ModelMutaLog.class.getName()).log(Level.SEVERE, null, ex);
        } catch (IOException ex) {
            Logger.getLogger(ModelMutaLog.class.getName()).log(Level.SEVERE, null, ex);
        }
        setRunning(false);
        setChanged();
        notifyObservers();
    }

    public void parseTestSuite(String file) throws FileNotFoundException, IOException {
        setRunning(true);
        setGlobalAction(GLOBAL_ACTION_LOAD_TEST_SUITE);
        setCurrentAction(CURRENT_ACTION_LOAD_TEST);
        testSuite = new TestSuite();
        BufferedReader in = new BufferedReader(new FileReader(file));
        String line;
        while ((line = in.readLine()) != null) {
            line = line.trim();

            StringTokenizer st = new StringTokenizer(line, " ");

            int token;
            Test test = new Test(st.countTokens());
            while (st.hasMoreTokens()) {
                token = Integer.parseInt(st.nextToken());
                test.addValue(token);
            }
            testSuite.add(test);

        }
        in.close();
        setRunning(false);
        setChanged();
        notifyObservers();
    }

    public void quit() {
        System.exit(0);
    }

    public CNFFormula getFormula() {
        return formula;
    }

    public TestSuite getTestSuite() {
        return testSuite;
    }

    public List<Mutant> getMutants() {
        return mutants;
    }

    public void createMutants(boolean literalOmission, boolean literalNegation, boolean clauseOmission, boolean clauseNegation, boolean operatorAnd, boolean operatorOr) {
        setRunning(true);
        setGlobalAction(GLOBAL_ACTION_CREATE_MUTANTS);

        setChanged();
        notifyObservers();
        mutants.clear();

        if (literalOmission) {
            setCurrentAction(CURRENT_ACTION_LITERAL_OMISSION_MUTANT);
            setChanged();
            notifyObservers();
            mutants.addAll(formula.getLiteralOmissionMutants());
        }

        if (literalNegation) {
            setCurrentAction(CURRENT_ACTION_LITERAL_NEGATION_MUTANT);
            setChanged();
            notifyObservers();
            mutants.addAll(formula.getLiteralNegationMutants());
        }

        if (clauseOmission) {
            setCurrentAction(CURRENT_ACTION_CLAUSE_OMISSION_MUTANT);
            setChanged();
            notifyObservers();
            mutants.addAll(formula.getClauseOmissionMutants());
        }

        if (clauseNegation) {
            setCurrentAction(CURRENT_ACTION_CLAUSE_NEGATION_MUTANT);
            setChanged();
            notifyObservers();
            mutants.addAll(formula.getClauseNegationMutants());
        }

        if (operatorAnd) {
            setCurrentAction(CURRENT_ACTION_OPERATOR_AND_MUTANT);
            setChanged();
            notifyObservers();
            mutants.addAll(formula.getOperatorReferenceAndMutants());
        }

        if (operatorOr) {
            setCurrentAction(CURRENT_ACTION_OPERATOR_OR_MUTANT);
            setChanged();
            notifyObservers();
            mutants.addAll(formula.getOperatorReferenceOrMutants());
        }

        setRunning(false);
        setChanged();
        notifyObservers();

    }

    public void evaluateMutationScore(boolean literalOmission, boolean literalNegation, boolean clauseOmission, boolean clauseNegation, boolean operatorAnd, boolean operatorOr, boolean print) {
        createMutants(literalOmission, literalNegation, clauseOmission, clauseNegation, operatorAnd, operatorOr);
        setRunning(true);
        setGlobalAction(GLOBAL_ACTION_EVAL_TEST_SUITE);
        setCurrentAction(CURRENT_ACTION_EVAL_MUTANTS);
        setChanged();
        notifyObservers();



        killed = 0;
        totalOk = 0;
        for (Mutant m : mutants) {
            boolean mutantOk = true;
            ISolver solver = SolverFactory.newDefault();
            solver.newVar(m.nVars);
            solver.setExpectedNumberOfClauses(m.nClauses());
            for (Clause c : m.getClauses()) {
                try {
                    IVecInt v = c.toVecInt();
                    solver.addClause(v);
                } catch (Exception ex) {
                    mutantOk = false;
                    break;
                }
            }

            if (mutantOk) {
                totalOk++;
                m.setInvalid(false);
                for (Test t : testSuite.getTests()) {
                    IVecInt test = t.toVecInt();


                    try {
                        if (!solver.isSatisfiable(test)) {
                            m.setKilled(true);
                            killed++;
                            System.out.println("(killed)  " + m.toStringMutant());
                            break;
                        } else {
                            System.out.println("(live)  " + m.toStringMutant());
                        }
                    } catch (TimeoutException ex) {
                        Logger.getLogger(ModelMutaLog.class.getName()).log(Level.SEVERE, null, ex);
                    }
                }
            } else {
                m.setInvalid(true);
                System.out.println("(invalid)  " + m.toStringMutant());
            }

        }

        if (print) {
            System.out.println("Score: " + this.killed + " / " + this.totalOk + " (" + (this.mutants.size() - totalOk) + " invalid)");
        }

        setRunning(false);
        setChanged();
        notifyObservers("score");

    }

    public int getKilled() {
        return killed;
    }

    public void setKilled(int killed) {
        this.killed = killed;
    }

    public int getTotalOk() {
        return totalOk;
    }

    public void setTotalOk(int totalOk) {
        this.totalOk = totalOk;
    }
}
