/*
 * Author : Christopher Henard (christopher.henard@uni.lu)
 * Date : 12/12/13
 * Copyright 2013 University of Luxembourg – Interdisciplinary Centre for Security Reliability and Trust (SnT)
 * All rights reserved
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package mutalog.gui.views;

import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.net.URL;
import java.util.Observable;
import java.util.Observer;
import javax.swing.ImageIcon;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import mutalog.core.ModelMutaLog;

public class ViewMenuBar extends JMenuBar implements Observer {

    private static final String FILE = "File";
    private static final String EXECUTE = "Execute";
    private static final String HELP = "Help";
    private static final String LOAD_FORMULA = "Load a CNF formula...";
    private static final String LOAD_TEST_SUITE = "Load a test suite...";
    private static final String QUIT = "Quit";
    public static final String COMPUTE_MUT_SCORE = "Evaluate the mutation score";
    public static final String STOP = "Stop the execution";
    public static final String DOC = "Documentation";
    public static final String ABOUT = "About MutaLog...";
    private ModelMutaLog model;
    private JMenu file, execute, help;
    private JMenuItem loadFormula, loadTestSuite, quit, coverage, stop, about, doc;
    private ImageIcon loadFormulaIcon, loadTestSuiteIcon, quitIcon,
            stopIcon, coverageIcon;
    private final URL urlLoadFormula = getClass().getResource("icons/load_formula.png");
    private final URL urlLoadTestSuite = getClass().getResource("icons/load_testsuite.png");
    private final URL urlQuit = getClass().getResource("icons/exit.png");
    private final URL urlCoverage = getClass().getResource("icons/coverage.png");
    private final URL urlStop = getClass().getResource("icons/stop.png");

    public ViewMenuBar(ModelMutaLog model) {
        super();
        this.model = model;
        model.addObserver(this);
        loadFormulaIcon = new ImageIcon(urlLoadFormula);
        loadTestSuiteIcon = new ImageIcon(urlLoadTestSuite);
        quitIcon = new ImageIcon(urlQuit);
        coverageIcon = new ImageIcon(urlCoverage);
        stopIcon = new ImageIcon(urlStop);
        loadFormula = new JMenuItem(LOAD_FORMULA, loadFormulaIcon);
        loadFormula.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F, ActionEvent.CTRL_MASK));
        loadTestSuite = new JMenuItem(LOAD_TEST_SUITE, loadTestSuiteIcon);
        loadTestSuite.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, ActionEvent.CTRL_MASK));
        quit = new JMenuItem(QUIT, quitIcon);
        quit.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, ActionEvent.CTRL_MASK));
        coverage = new JMenuItem(COMPUTE_MUT_SCORE, coverageIcon);
        coverage.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G, ActionEvent.ALT_MASK));
        stop = new JMenuItem(STOP, stopIcon);
        stop.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_K, ActionEvent.ALT_MASK));
        doc = new JMenuItem(DOC);
        doc.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, ActionEvent.ALT_MASK));
        about = new JMenuItem(ABOUT);
        about.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, ActionEvent.ALT_MASK));

        file = new JMenu(FILE);
        file.setMnemonic(KeyEvent.VK_F);
        file.add(loadFormula);
        file.add(loadTestSuite);
        file.addSeparator();
        file.add(quit);
        add(file);
        execute = new JMenu(EXECUTE);
        execute.setMnemonic(KeyEvent.VK_E);
        execute.add(coverage);
        execute.add(coverage);
        execute.add(stop);
        execute.addSeparator();
        add(execute);
        help = new JMenu(HELP);
        help.setMnemonic(KeyEvent.VK_H);
        help.add(doc);
        help.addSeparator();
        help.add(about);
        add(help);
        loadTestSuite.setEnabled(model.getFormula() != null);
        coverage.setEnabled(model.getTestSuite() != null);
        stop.setEnabled(model.isRunning());
        execute.setEnabled(stop.isEnabled() || coverage.isEnabled());
    }

    public JMenuItem getLoadFormula() {
        return loadFormula;
    }

    public JMenuItem getLoadTestSuite() {
        return loadTestSuite;
    }

    public JMenuItem getQuit() {
        return quit;
    }

    public JMenuItem getCoverage() {
        return coverage;
    }

    public JMenuItem getStop() {
        return stop;
    }

    public JMenuItem getAbout() {
        return about;
    }

    public JMenuItem getDoc() {
        return doc;
    }

    @Override
    public void update(Observable o, Object arg) {
        setEnabled();
    }

    private void setEnabled() {
        Runnable code = new Runnable() {
            @Override
            public void run() {
                loadTestSuite.setEnabled(model.getFormula() != null);
                coverage.setEnabled(model.getTestSuite() != null);
                stop.setEnabled(model.isRunning());
                execute.setEnabled(stop.isEnabled() || coverage.isEnabled());
            }
        };
        if (SwingUtilities.isEventDispatchThread()) {
            code.run();
        } else {
            SwingUtilities.invokeLater(code);
        }
    }
}
