/*
 * Author : Christopher Henard (christopher.henard@uni.lu)
 * Date : 12/12/13
 * Copyright 2013 University of Luxembourg – Interdisciplinary Centre for Security Reliability and Trust (SnT)
 * All rights reserved
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package mutalog.gui.views;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridLayout;
import java.io.File;
import java.net.URL;
import java.util.Observable;
import java.util.Observer;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.SwingUtilities;
import javax.swing.filechooser.FileNameExtensionFilter;
import mutalog.core.ModelMutaLog;
import mutalog.gui.views.controllers.ControllerCloseAbout;
import mutalog.gui.views.controllers.ControllerCoverage;
import mutalog.gui.views.controllers.ControllerDisplayAbout;
import mutalog.gui.views.controllers.ControllerDisplayDocumentation;
import mutalog.gui.views.controllers.ControllerLoadFormula;
import mutalog.gui.views.controllers.ControllerLoadTestSuite;
import mutalog.gui.views.controllers.ControllerQuit;
import mutalog.gui.views.controllers.ControllerRunCoverage;

public class ViewMutaLog extends JFrame implements Observer {

    public static final Color BLUE_COLOR = new Color(232, 239, 247);
    public static final int D_WIDTH = 900, D_HEIGHT = 600;
    public static final String TITLE = "MutaLog - a Tool for Mutating Logic Formulas";
    public static final String MUTATION_SCORE = "Mutation score";
    private static final String QUIT_MESSAGE = "Quit MutaLog will interrupt any running task. Do you want to continue ?";
    private static final String QUIT_TITLE = "Quit MutaLog";
    private static final String FILE_CHOOSER_FORMULA_TITLE = "Load a CNF formula";
    private static final String FILE_CHOOSER_TEST_SUITE_TITLE = "Load a test suite";
    private static final String ERROR = "Error";
    private static final String NO_MUTANTS_SELECTED = "No mutant selected";
    private final FileNameExtensionFilter formulaFileFilter = new FileNameExtensionFilter("DIMACS format (.dimacs)", "dimacs");
    private final URL BACKGROUND_URL = getClass().getResource("icons/mutalog.png");
    private ModelMutaLog model;
    private ControllerDisplayAbout controllerDisplayAbout;
    private ControllerCloseAbout controllerCloseAbout;
    private ControllerQuit controllerQuit;
    private ControllerLoadFormula controllerLoadFormula;
    private ControllerLoadTestSuite controllerLoadTestSuite;
    private ControllerDisplayDocumentation controllerDisplayDocumentation;
    private ControllerCoverage controllerCoverage;
    private ControllerRunCoverage controllerRunCoverage;
    private ViewMenuBar viewMenuBar;
    private ViewStatusBar viewStatusBar;
    private ViewAboutWindow viewAboutWindow;
    private ViewFormula viewFormula;
    private ViewTestSuite viewTestSuite;
    private ViewMutantsConfiguration viewMutantsConfiguration;
    private ViewMutants viewMutants;
    private ViewToolBar viewToolBar;
    private ViewDocumentation viewDocumentation;
    private JPanel content, background;
    private JFileChooser formulaChooser, testSuiteChooser;
    private JSplitPane splitPane;

    public ViewMutaLog() {
        super(TITLE);
        model = new ModelMutaLog();
        model.addObserver(this);
        new Thread(new Runnable() {
            @Override
            public void run() {
                viewDocumentation = ViewDocumentation.getInstance();
                viewDocumentation.setLocationRelativeTo(ViewMutaLog.this);
            }
        }).start();
        viewAboutWindow = new ViewAboutWindow(this);
        viewStatusBar = new ViewStatusBar(model);
        viewMenuBar = new ViewMenuBar(model);
        viewFormula = new ViewFormula(model);
        viewTestSuite = new ViewTestSuite(model);
        viewMutantsConfiguration = new ViewMutantsConfiguration(model);
        viewMutants = new ViewMutants(model);
        viewToolBar = new ViewToolBar(model);

        controllerQuit = new ControllerQuit(model, this);
        controllerLoadFormula = new ControllerLoadFormula(model, this);
        controllerLoadTestSuite = new ControllerLoadTestSuite(model, this);
        controllerDisplayAbout = new ControllerDisplayAbout(this);
        controllerCloseAbout = new ControllerCloseAbout(this);
        controllerDisplayDocumentation = new ControllerDisplayDocumentation(this);
        controllerCoverage = new ControllerCoverage(model, this);
        controllerRunCoverage = new ControllerRunCoverage(model, this, viewMutantsConfiguration);

        viewAboutWindow.getCloseButton().addActionListener(controllerCloseAbout);
        viewMenuBar.getQuit().addActionListener(controllerQuit);
        viewMenuBar.getLoadFormula().addActionListener(controllerLoadFormula);
        viewMenuBar.getLoadTestSuite().addActionListener(controllerLoadTestSuite);
        viewMenuBar.getAbout().addActionListener(controllerDisplayAbout);
        viewMenuBar.getDoc().addActionListener(controllerDisplayDocumentation);
        viewMenuBar.getCoverage().addActionListener(controllerCoverage);


        viewToolBar.getQuit().addActionListener(controllerQuit);
        viewToolBar.getLoadFormula().addActionListener(controllerLoadFormula);
        viewToolBar.getLoadTestSuite().addActionListener(controllerLoadTestSuite);
        viewToolBar.getCoverage().addActionListener(controllerCoverage);

        viewMutantsConfiguration.getOk().addActionListener(controllerRunCoverage);

        splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        setJMenuBar(viewMenuBar);


        formulaChooser = new JFileChooser();
        formulaChooser.setMultiSelectionEnabled(false);
        formulaChooser.setAcceptAllFileFilterUsed(false);
        formulaChooser.addChoosableFileFilter(formulaFileFilter);
        formulaChooser.setDialogTitle(FILE_CHOOSER_FORMULA_TITLE);

        testSuiteChooser = new JFileChooser();
        testSuiteChooser.setMultiSelectionEnabled(false);
        testSuiteChooser.setAcceptAllFileFilterUsed(false);
        testSuiteChooser.setDialogTitle(FILE_CHOOSER_TEST_SUITE_TITLE);

        ViewMutaLog.this.add(viewStatusBar, BorderLayout.SOUTH);
        content = new JPanel(new BorderLayout());

        background = new JPanel(new GridLayout(1, 1));
        background.setBorder(BorderFactory.createMatteBorder(1, 0, 1, 0, new Color(172, 168, 153)));
        background.setBackground(Color.white);
        background.add(new JLabel(new ImageIcon(BACKGROUND_URL)), BorderLayout.CENTER);
        add(background, BorderLayout.CENTER);
        add(viewToolBar, BorderLayout.PAGE_START);
        setSize(D_WIDTH, D_HEIGHT);
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        setLocationRelativeTo(getParent());
        setVisible(true);
    }

    public boolean getQuitConfirmation() {
        return JOptionPane.showConfirmDialog(this, QUIT_MESSAGE, QUIT_TITLE, JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION;
    }

    @Override
    public void update(Observable o, final Object arg) {
        Runnable code = new Runnable() {
            @Override
            public void run() {

                content.removeAll();
                if (model.getFormula() != null) {
                    remove(background);
                    add(content, BorderLayout.CENTER);
                    viewStatusBar.addSeparator();
                    content.add(viewFormula, BorderLayout.NORTH);

                }

                if (model.getTestSuite() != null) {
                    content.add(viewTestSuite, BorderLayout.CENTER);

                    if (model.getMutants().size() > 0) {
                        splitPane.setLeftComponent(viewTestSuite);
                        splitPane.setRightComponent(viewMutants);
                        content.add(splitPane, BorderLayout.CENTER);
                        //content.add(viewMutants, BorderLayout.EAST);
                    }

                } else {
//                    content.remove(viewTestSuite);
//                     content.remove(viewMutants);
                    splitPane.removeAll();
                    content.remove(splitPane);
                }

                if (arg != null) {
                    displayScore();
                }

                validate();
                SwingUtilities.updateComponentTreeUI(ViewMutaLog.this);
            }
        };
        if (SwingUtilities.isEventDispatchThread()) {
            code.run();
        } else {
            SwingUtilities.invokeLater(code);
        }
    }

    public void displayViewAboutWindow(final boolean display) {
        Runnable code = new Runnable() {
            @Override
            public void run() {
                viewAboutWindow.setVisible(display);
            }
        };
        if (SwingUtilities.isEventDispatchThread()) {
            code.run();
        } else {
            SwingUtilities.invokeLater(code);
        }
    }

    public File displayFormulaChooser() {


        if (formulaChooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
            return formulaChooser.getSelectedFile();
        }

        return null;
    }

    public File displayTestSuiteChooser() {


        if (testSuiteChooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
            return testSuiteChooser.getSelectedFile();
        }

        return null;
    }

    public void displayDocumentation() {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                viewDocumentation.setVisible(true);
            }
        });
    }

    public void displayMutantsConfiguration() {
        Runnable code = new Runnable() {
            @Override
            public void run() {
                viewMutantsConfiguration.setLocationRelativeTo(ViewMutaLog.this);
                viewMutantsConfiguration.setVisible(true);
            }
        };
        if (SwingUtilities.isEventDispatchThread()) {
            code.run();
        } else {
            SwingUtilities.invokeLater(code);
        }

    }

    public void displayErrorMessageNoMutants() {
        Runnable code = new Runnable() {
            @Override
            public void run() {
                JOptionPane.showMessageDialog(ViewMutaLog.this, NO_MUTANTS_SELECTED, ERROR, JOptionPane.ERROR_MESSAGE);
            }
        };
        if (SwingUtilities.isEventDispatchThread()) {
            code.run();
        } else {
            SwingUtilities.invokeLater(code);
        }
    }

    public void displayScore() {
        Runnable code = new Runnable() {
            @Override
            public void run() {
                JOptionPane.showMessageDialog(ViewMutaLog.this, model.getKilled() + " / " + model.getTotalOk() + " (" + (model.getMutants().size() - model.getTotalOk()) + " invalid mutants)", MUTATION_SCORE, JOptionPane.INFORMATION_MESSAGE);
            }
        };
        if (SwingUtilities.isEventDispatchThread()) {
            code.run();
        } else {
            SwingUtilities.invokeLater(code);
        }
    }
}
