
package multiobj;

/*
 * Author : Christopher Henard (christopher.henard@uni.lu)
 * Date : 23/02/2013
 * Copyright 2012 University of Luxembourg – Interdisciplinary Centre for Security Reliability and Trust (SnT)
 * All rights reserved
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;
import com.beust.jcommander.Parameters;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.StringTokenizer;
import org.sat4j.core.VecInt;
import org.sat4j.minisat.SolverFactory;
import org.sat4j.minisat.core.Solver;
import org.sat4j.minisat.orders.RandomLiteralSelectionStrategy;
import org.sat4j.minisat.orders.RandomWalkDecorator;
import org.sat4j.minisat.orders.VarOrderHeap;
import org.sat4j.reader.DimacsReader;
import org.sat4j.specs.ISolver;
import org.sat4j.specs.IVecInt;
import org.sat4j.tools.ModelIterator;
import splar.core.constraints.CNFClause;
import splar.core.constraints.CNFFormula;
import splar.core.fm.FeatureModel;
import splar.core.fm.XMLFeatureModel;
import splar.plugins.reasoners.sat.sat4j.FMReasoningWithSAT;

/**
 *
 * @author chris
 */
public class Multiobj {

    private CommandLineParser parser;
    private double w1, w2, w3;
    private double maxProductCost;
    private int popSize;
    private int runs;
    private int minProds = 1;
    private int maxIndSize;
    private double maxPairs;
    private String fm;
    private String fcost;
    private String outFile;
    private String outIndiv;
    HashMap<Integer, Integer> costs;
    private int elitism;
    private double mutateProb;

    public Multiobj(String[] args) {

        try {
            parser = new CommandLineParser(args, "Multiobj");

            if (args.length == 0) {
                throw new ParameterException("No arguments");
            }

            parser.parseArgs();
            String command = parser.getCommandName();

            if (command.equals(CommandLineParser.TO_DIMACS)) {
                String splot = parser.getCommandToDimacs().fmFile;
                String dimacs = splot.replaceAll("xml", "dimacs");
                splotToDimacs(dimacs, splot);
            } else if (command.equals(CommandLineParser.MOGA)) {

                w1 = parser.getCommandMOGA().w1;
                w2 = parser.getCommandMOGA().w2;
                w3 = parser.getCommandMOGA().w3;
                fm = parser.getCommandMOGA().fmFile;
                fcost = parser.getCommandMOGA().costFile;
                runs = parser.getCommandMOGA().runs;
                maxIndSize = parser.getCommandMOGA().maxProds;
                popSize = parser.getCommandMOGA().popSize;
                maxPairs = parser.getCommandMOGA().pairs;
                outFile = parser.getCommandMOGA().out;
                outIndiv = parser.getCommandMOGA().outIndiv;
                costs = loadCost(fcost);
                elitism = parser.getCommandMOGA().elitism;
                maxProductCost = 0;
                for (Integer i : costs.keySet()) {
                    maxProductCost += costs.get(i);
                }

                Individual sol = multiObjectiveGeneticAlgorithm();
                writeProds(outIndiv, sol);

            } else if (command.equals(CommandLineParser.PAIRS)) {
                fm = parser.getCommandPairs().fmFile;
                System.out.println(getFMNumberOfPairs());
            }

        } catch (Exception e) {
            System.out.println("ERROR: " + e.getMessage());
            parser.printUsage();
        }
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        new Multiobj(args);
    }

    public int getFMNumberOfPairs() throws Exception {
        Set<TSet> pairs = new HashSet<TSet>();
        ISolver dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
        DimacsReader dr = new DimacsReader(dimacsSolver);
        dr.parseInstance(new FileReader(fm));
        Solver solver = (Solver) dimacsSolver;

        ArrayList<Integer> featuresIntList = new ArrayList<Integer>();
        BufferedReader in = new BufferedReader(new FileReader(fm));
        String line;
        int n = 0;
        while ((line = in.readLine()) != null && line.startsWith("c")) {
            StringTokenizer st = new StringTokenizer(line.trim(), " ");
            st.nextToken();
            n++;
            String sFeature = st.nextToken().replace('$', ' ').trim();
            int feature = Integer.parseInt(sFeature);
            if (n != feature) {
                throw new Exception("Incorrect dimacs file, missing feature number " + n + " ?");
            }
            featuresIntList.add(feature);
        }
        in.close();

        List<Integer> extendedFeatures = new ArrayList<Integer>(featuresIntList.size() * 2);

        for (Integer i : featuresIntList) {
            extendedFeatures.add(i);
            extendedFeatures.add(-i);
        }

        int size = extendedFeatures.size();

        nCk(size, 2, pairs, extendedFeatures, true, solver);

        return pairs.size();
    }
    
    public void nCk(int n, int k, Set<TSet> tsets, List<Integer> featuresList, boolean checkValid, Solver solver) throws Exception {
        int[] a = new int[k];
        nCkH(n, k, 0, a, k, tsets, featuresList, checkValid, solver);
    }
    
    public void nCkH(int n, int loopno, int ini, int[] a, int k, Set<TSet> tsets, List<Integer> featuresList, boolean checkValid, Solver solver) throws Exception {
        
        if (k == 0) {
            return;
        }
        
        int i;
        loopno--;
        if (loopno < 0) {
            a[k - 1] = ini - 1;
            TSet p = new TSet();
            for (i = 0; i < k; i++) {
                p.add(featuresList.get(a[i]));
            }
            
            if (checkValid) {
                IVecInt prod = new VecInt(p.getSize());
                
                for (Integer in : p.getVals()) {
                    prod.push(in);
                }
                
                if (solver.isSatisfiable(prod)) {
                    tsets.add(p);
                }
            } else {
                tsets.add(p);
            }
            return;
            
        }
        for (i = ini; i <= n - loopno - 1; i++) {
            a[k - 1 - loopno] = i;
            nCkH(n, loopno, i + 1, a, k, tsets, featuresList, checkValid, solver);
        }
        
        
    }

    public void writeProds(String file, Individual i) throws Exception {
        BufferedWriter out = new BufferedWriter(new FileWriter(file));

        for (IVecInt v : i.getProds()) {
            for (int j = 0; j < v.size(); j++) {
                out.write(v.get(j) + " ");

            }
            out.newLine();
        }
        out.close();
    }

    public void writeRes(String file, Individual i) throws Exception {
        BufferedWriter out = new BufferedWriter(new FileWriter(file));
        out.write("fitness:" + i.getFitness());
        out.newLine();
        out.write("o1:" + i.getO1());
        out.newLine();
        out.write("o2:" + i.getO2());
        out.newLine();
        out.write("o3:" + i.getO3());
        out.newLine();
        out.write("pairs:" + i.getPairs());
        out.newLine();
        out.write("maxpairs:" + i.getMaxpairs());
        out.newLine();
        out.write("cost:" + i.getCost());
        out.newLine();
        out.write("max cost:" + i.getMaxcost());
        out.newLine();
        out.write("nbprods:" + i.getNbprods());
        out.newLine();
        out.close();
    }

    // TO DO : create command line for this to be usable by someone else
    public Individual getRandomIndividualSameCoverage(int pairs) throws Exception {


        List<IVecInt> rands = new ArrayList<IVecInt>();

        do {
            rands.add(getDissimilarConfigs(1, fm).get(0));
        } while (getPairwiseCoverage(rands) < pairs);

        Individual i = new Individual(rands);
        evaluateFitness(i);

        return i;
    }

    public int getPairwiseCoverage(List<IVecInt> prods) {
        Set<TSet> pairs = new HashSet<TSet>();
        for (IVecInt v : prods) {

            for (int i = 0; i < v.size(); i++) {
                for (int j = 0; j < v.size(); j++) {
                    if (j > i) {
                        pairs.add(new TSet(new int[]{v.get(i), v.get(j)}));
                    }

                }

            }
        }
        return pairs.size();
    }

    // TO DO : create command line for this to be usable by someone else
    public Individual getRandomIndividualSameNumberProducts(int nbProds) throws Exception {


        List<IVecInt> rands = new ArrayList<IVecInt>();

        while (rands.size() < nbProds) {
            rands.add(getDissimilarConfigs(1, fm).get(0));
        }

        Individual i = new Individual(rands);
        evaluateFitness(i);

        return i;
    }

    public Individual multiObjectiveGeneticAlgorithm() throws Exception {


        List<Individual> indivs = new ArrayList<Individual>();
        double totalFitness = 0;


        Random r = new Random();
        for (int i = 1; i <= 100; i++) {
            int size = r.nextInt(maxIndSize - 1) + minProds;
            List<IVecInt> prods = getDissimilarConfigs(size, fm);
            Individual indiv = new Individual(prods);
            evaluateFitness(indiv);
            totalFitness += indiv.getFitness();
            indivs.add(indiv);

        }

        Collections.sort(indivs);


//        BufferedWriter out = new BufferedWriter(new FileWriter(outFile));

//        out.write("" + indivs.get(0).getFitness() + ";" + indivs.get(0).getO1() + ";" + indivs.get(0).getO2() + ";" + indivs.get(0).getO3() + ";" + indivs.get(0).getPairs() + ";" + indivs.get(0).getNbprods() + ";" + indivs.get(0).getCost() + ";" + indivs.get(0).getMaxcost());
//        out.newLine();
        int nGen = 1;

        while (nGen <= runs) {

            System.out.println("Generation " + nGen);

            List<Individual> pop = new ArrayList<Individual>();
            /* elitism */


            for (int i = 0; i < elitism; i++) {
                pop.add(indivs.get(i));

            }

            int size = r.nextInt(maxIndSize - 1) + minProds;
            List<IVecInt> prods = getDissimilarConfigs(size, fm);

            Individual rand = new Individual(prods);
            evaluateFitness(rand);
            pop.add(rand);

            //System.out.println("fit->" + indivs.get(0).getFitness() + ", o1: " + indivs.get(0).getO1() + ", o2:" + indivs.get(0).getO2() + ", o3:" + indivs.get(0).getO3() + ",  size: " + indivs.get(0).getProds().size());

            while (pop.size() < popSize) {
                int rnd = (int) (Math.random() * totalFitness);

                int idx = 1;
                for (idx = 0; idx < popSize && rnd > 0; idx++) {
                    rnd -= indivs.get(idx).getFitness();
                }
                if (idx == 0) {
                    idx = 1;
                }

                Individual p1 = indivs.get(idx - 1);



                rnd = (int) (Math.random() * totalFitness);

                idx = 1;
                for (idx = 0; idx < popSize && rnd > 0; idx++) {
                    rnd -= indivs.get(idx).getFitness();
                }

                if (idx == 0) {
                    idx = 1;
                }

                Individual p2 = indivs.get(idx - 1);

                List<List<IVecInt>> offsprings = cross(p1.getProds(), p2.getProds(), r);



                double prob = Math.random() * 100.0;

                if (prob <= mutateProb) {

                    mutate(offsprings.get(0), r, fm);
                }

                prob = Math.random() * 100.0;

                if (prob <= mutateProb) {
                    mutate(offsprings.get(1), r, fm);
                }



                Individual off1 = new Individual(offsprings.get(0));
                Individual off2 = new Individual(offsprings.get(1));



                evaluateFitness(off1);
                evaluateFitness(off2);

                pop.add(off1);
                pop.add(off2);
            }

            indivs = pop;




            Collections.sort(indivs);

            while (indivs.size() < 100) {
                indivs.remove(indivs.size() - 1);
            }

            totalFitness = 0;
            for (Individual i : indivs) {
                totalFitness += i.getFitness();
            }

            nGen++;

//            out.write("" + indivs.get(0).getFitness() + ";" + indivs.get(0).getO1() + ";" + indivs.get(0).getO2() + ";" + indivs.get(0).getO3() + ";" + indivs.get(0).getPairs() + ";" + indivs.get(0).getNbprods() + ";" + indivs.get(0).getCost() + ";" + indivs.get(0).getMaxcost());
//            out.newLine();

            //System.out.println("fit->" + indivs.get(0).getFitness()+ ", o1: "+ indivs.get(0).getO1() + ", o2:" + indivs.get(0).getO2() + ", o3:"+ indivs.get(0).getO3()+ ", size: "+indivs.get(0).getProds().size());

        }
//        out.close();

        writeRes(outFile, indivs.get(0));
        return indivs.get(0);
    }

    public void mutate(List<IVecInt> prods, Random r, String fm) throws Exception {

        int prodIndex = r.nextInt(prods.size());
        prods.remove(prodIndex);
        prods.add(getDissimilarConfigs(1, fm).get(0));
    }

    public List<List<IVecInt>> cross(List<IVecInt> p1, List<IVecInt> p2, Random r) {



        List<List<IVecInt>> offsprings = new ArrayList<List<IVecInt>>();

        List<IVecInt> o1 = new ArrayList<IVecInt>(p1);
        List<IVecInt> o2 = new ArrayList<IVecInt>(p2);


        if (p1.size() <= p2.size()) {


            int c = r.nextInt(p1.size());

            int n = 0;

            while (n < c) {

                int i = r.nextInt(p1.size());
                int j = r.nextInt(p2.size());

                IVecInt tmp = o1.get(i);

                o1.set(i, o2.get(j));
                o2.set(j, tmp);
                n++;
            }
        } else {
            int c = r.nextInt(p2.size());

            int n = 0;

            while (n < c) {

                int i = r.nextInt(p2.size());
                int j = r.nextInt(p1.size());

                IVecInt tmp = o2.get(i);

                o2.set(i, o1.get(j));
                o1.set(j, tmp);
                n++;
            }
        }

        offsprings.add(o1);
        offsprings.add(o2);

        return offsprings;

    }

    public int getCost(List<IVecInt> products) {
        int cost = 0;
        for (IVecInt p : products) {
            for (int i = 0; i < p.size(); i++) {
                if (p.get(i) > 0) {

                    cost += costs.get(p.get(i)) == null ? 0 : costs.get(p.get(i));
                }

            }
        }
        return cost;
    }

    public HashMap<Integer, Integer> loadCost(String filename) throws Exception {
        BufferedReader in = new BufferedReader(new FileReader(filename));
        HashMap<Integer, Integer> costs = new HashMap<Integer, Integer>();
        String line;
        while ((line = in.readLine()) != null) {
            StringTokenizer st = new StringTokenizer(line, ":");
            costs.put(Integer.parseInt(st.nextToken()), Integer.parseInt(st.nextToken()));
        }
        in.close();
        return costs;

    }

    public void evaluateFitness(Individual indiv) {
        double o1, o2, o3;

        List<IVecInt> products = indiv.getProds();


        double n = products.size();
        Set<TSet> pairs = new HashSet<TSet>();
        List<IVecInt> prods = indiv.getProds();
        for (IVecInt v : prods) {

            for (int i = 0; i < v.size(); i++) {
                for (int j = 0; j < v.size(); j++) {
                    if (j > i) {
                        pairs.add(new TSet(new int[]{v.get(i), v.get(j)}));
                    }

                }

            }
        }

        int nf = products.get(0).size();

        o1 = (-pairs.size() + maxPairs) / (-(nf * (nf - 1) / 2) + maxPairs);

        o2 = (n - minProds) / (maxIndSize - minProds);
        double cost = getCost(products);


        double maxCost = maxProductCost * n;

        o3 = (double) cost / maxCost;

        double fit = o1 * w1 + o2 * w2 + o3 * w3;

        indiv.setFitness(fit);
        indiv.setO1(o1);
        indiv.setO2(o2);
        indiv.setO3(o3);
        indiv.setMaxpairs(maxPairs);
        indiv.setPairs(pairs.size());
        indiv.setMaxcost(maxCost);
        indiv.setCost(cost);
        indiv.setNbprods(n);

    }

    public List<IVecInt> getDissimilarConfigs(int count, String dimacsFM) throws Exception {

        ISolver dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
        DimacsReader dr = new DimacsReader(dimacsSolver);
        dr.parseInstance(new FileReader(dimacsFM));
        Solver solver = (Solver) dimacsSolver;
        solver.setTimeout(1000);
        solver.setOrder(new RandomWalkDecorator(new VarOrderHeap(new RandomLiteralSelectionStrategy()), 1));
        ISolver solverIterator = new ModelIterator(solver);
        solverIterator.setTimeoutMs(150000);

        List<IVecInt> products = new ArrayList<IVecInt>(count);


        while (products.size() < count) {

            try {
                if (solverIterator.isSatisfiable()) {

                    int[] vec = solverIterator.model();


                    IVecInt vect = toVec(vec);

                    if (!products.contains(vect)) {
                        products.add(vect);
                    }


                } else {

                    dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                    dr = new DimacsReader(dimacsSolver);
                    dr.parseInstance(new FileReader(dimacsFM));
                    solver = (Solver) dimacsSolver;
                    solver.setTimeout(1000);
                    solver.setOrder(new RandomWalkDecorator(new VarOrderHeap(new RandomLiteralSelectionStrategy()), 1));

                    solverIterator = new ModelIterator(solver);
                    solverIterator.setTimeoutMs(150000);
                }

            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return products;
    }

    public IVecInt toVec(int[] vec) {
        IVecInt vect = new VecInt(vec.length);
        for (int i = 0; i < vec.length; i++) {
            vect.push(vec[i]);

        }
        return vect;
    }

    public void splotToDimacs(String dimacsPath, String splotPath) throws Exception {

        FeatureModel splotFM = new XMLFeatureModel(splotPath, XMLFeatureModel.USE_VARIABLE_NAME_AS_ID);
        splotFM.loadModel();
        int SATtimeout = 60000;
        FMReasoningWithSAT reasoner = new FMReasoningWithSAT("MiniSAT", splotFM, SATtimeout);

        reasoner.init();


        File file = new File(dimacsPath);

        BufferedWriter out = new BufferedWriter(new FileWriter(file));

        for (int i = 1; i <= splotFM.countFeatures(); i++) {
            out.write("c " + i + " " + reasoner.getVariableName(i));
            out.newLine();

        }
        CNFFormula formula = splotFM.FM2CNF();

        out.write("p cnf " + splotFM.countFeatures() + " " + formula.getClauses().size());
        out.newLine();
        for (CNFClause clause : formula.getClauses()) {
            for (int i = 0; i < clause.getLiterals().size(); i++) {
                int signal = clause.getLiterals().get(i).isPositive() ? 1 : -1;
                int varID = reasoner.getVariableIndex(clause.getLiterals().get(i).getVariable().getID());
                out.write("" + (signal * varID) + " ");

                if (i == clause.getLiterals().size() - 1) {
                    out.write("0");
                    out.newLine();
                }
            }
        }
        out.close();
    }

    private class CommandLineParser {

        private JCommander jCommander;
        private MOGA commandMOGA;
        private ToDimacs commandToDimacs;
        private Pairs commandPairs;
        public static final String MOGA = "moga";
        public static final String TO_DIMACS = "todimacs";
        public static final String PAIRS = "numpairs";
        private String[] args;

        public CommandLineParser(String[] args, String programName) {
            this.args = args;
            commandMOGA = new MOGA();
            commandToDimacs = new ToDimacs();
            commandPairs = new Pairs();
            jCommander = new JCommander();
            
            jCommander.addCommand(TO_DIMACS, commandToDimacs);
            jCommander.addCommand(PAIRS, commandPairs);
            jCommander.addCommand(MOGA, commandMOGA);
            jCommander.setProgramName("java -jar " + programName + ".jar");
        }

        public void parseArgs() {
            jCommander.parse(args);
        }

        public void printUsage() {
            jCommander.usage();
        }

        public String getCommandName() {
            return jCommander.getParsedCommand();
        }

        public MOGA getCommandMOGA() {
            return commandMOGA;
        }

        public ToDimacs getCommandToDimacs() {
            return commandToDimacs;
        }

        public Pairs getCommandPairs() {
            return commandPairs;
        }

        @Parameters(commandDescription = "Run the multi-objective genetic algorithm. ")
        private class MOGA {

            @Parameter(names = "-fm", description = "Feature model (DIMACS format)", required = true)
            private String fmFile;
            @Parameter(names = "-cost", description = "Cost file (following the format featureNumber:cost)", required = true)
            private String costFile;
            @Parameter(names = "-outInfo", description = "Output file to store the information of each run", required = true)
            private String out;
            @Parameter(names = "-outSol", description = "Output file to store the products of the Individual solution to the problem", required = true)
            private String outIndiv;
            @Parameter(names = "-pairs", description = "Number of valid pairs of the feature model", required = true)
            private int pairs;
            @Parameter(names = "-runs", description = "Number of runs to perform")
            private int runs = 500;
            @Parameter(names = "-w1", description = "Weight for objective F1")
            private double w1 = 0.5;
            @Parameter(names = "-w2", description = "Weight for objective F2")
            private double w2 = 0.25;
            @Parameter(names = "-w3", description = "Weight for objective F3")
            private double w3 = 0.25;
            @Parameter(names = "-maxIndSize", description = "Maximum number of products for an Individual, i.e. a solution to the problem")
            private int maxProds = 100;
            @Parameter(names = "-popSize", description = "Maximum size for the population (set of solutions)")
            private int popSize = 100;
            @Parameter(names = "-elitism", description = "Number individuals to keep for the next generation (elitism), should be lower than the population size")
            private int elitism = 5;
            @Parameter(names = "-mutationProb", description = "Probability to mutate an offspring (in percentage)")
            private double mutateProb = 5.0;
        }

        @Parameters(commandDescription = "Convert a Feature Model from the SPLOT (.xml) format to the DIMACS (.dimacs) format. ")
        private class ToDimacs {

            @Parameter(names = "-fm", description = "Feature model in the SPLOT (.xml) format)", required = true)
            private String fmFile;
        }

        @Parameters(commandDescription = "Compute the number of valid pairs of a given DIMACS Feature model (.dimacs format). ")
        private class Pairs {

            @Parameter(names = "-fm", description = "Feature model in the DIMACS (.dimacs) format", required = true)
            private String fmFile;
        }
    }
}
