/*
 * Author : Christopher Henard (christopher.henard@uni.lu)
 * Date : 21/05/2012
 * Copyright 2012 University of Luxembourg – Interdisciplinary Centre for Security Reliability and Trust (SnT)
 * All rights reserved
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package spl;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;
import com.beust.jcommander.Parameters;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.sat4j.core.VecInt;
import org.sat4j.minisat.SolverFactory;
import org.sat4j.minisat.core.IOrder;
import org.sat4j.minisat.core.Solver;
import org.sat4j.minisat.orders.RandomLiteralSelectionStrategy;
import org.sat4j.minisat.orders.RandomWalkDecorator;
import org.sat4j.minisat.orders.VarOrderHeap;
import org.sat4j.reader.DimacsReader;
import org.sat4j.specs.ISolver;
import org.sat4j.specs.IVecInt;
import org.sat4j.specs.TimeoutException;
import org.sat4j.tools.ModelIterator;
import spl.fm.TSet;
import spl.fm.Product;
import spl.techniques.DistancesUtil;
import spl.techniques.RandomTechnique;
import spl.techniques.SimilarityTechnique;
import spl.techniques.ga.GA;
import spl.techniques.ga.Individual;
import splar.core.fm.FeatureModel;
import splar.core.fm.XMLFeatureModel;
import splar.core.heuristics.FTPreOrderSortedECTraversalHeuristic;
import splar.core.heuristics.VariableOrderingHeuristic;
import splar.core.heuristics.VariableOrderingHeuristicsManager;
import splar.plugins.reasoners.bdd.javabdd.FMReasoningWithBDD;
import splar.plugins.reasoners.bdd.javabdd.ReasoningWithBDD;
import splar.plugins.reasoners.sat.sat4j.FMReasoningWithSAT;
import splar.plugins.reasoners.sat.sat4j.ReasoningWithSAT;
import org.apache.commons.math3.util.ArithmeticUtils;

public class SPL {

    private static Random randomGenerator = new Random();
    private FeatureModel fm;
    private ReasoningWithSAT reasonerSAT;
    private ISolver solverIterator, dimacsSolver;
    private final IOrder order = new RandomWalkDecorator(new VarOrderHeap(new RandomLiteralSelectionStrategy()), 1);
    private CommandLineParser parser;
    private static SPL instance = null;
    private final int SATtimeout = 1000;
    private final long iteratorTimeout = 150000;
    private boolean dimacs;
    private String dimacsFile;
    private boolean predictable;

    protected SPL() {
    }

    public static SPL getInstance() {
        if (instance == null) {
            instance = new SPL();
        }
        return instance;
    }

    private int largestV(int a, int b, double x) {
        int v = a - 1;

        while (getBinomCoeff(v, b) > x) {
            v--;
        }

        return v;
    } // LargestV()

    public double getBinomCoeff(int n, int k) {
        if (k > n) {
            return 0.0;
        } else if (n == k || k == 0) {
            return 1.0;
        } else {
            return ArithmeticUtils.binomialCoefficientDouble(n, k);
        }
    }

    public TSet getITSet(int n, int k, double m, List<Integer> featuresList) {

        double total = getBinomCoeff(n, k);
        if (m >= total) {
            m = total - 1.0;
        }
        TSet tSet = new TSet();
        int a = n;
        int b = k;
        double x = (total - 1.0) - m;  // x is the "dual" of m

        for (int i = 0; i < k; i++) {
            a = largestV(a, b, x);          // largest value v, where v < a and vCb < x
            x = x - getBinomCoeff(a, b);
            b = b - 1;
            tSet.add(featuresList.get(n - 1 - a));
        }

        return tSet;
    }

    public void estimateValidTSets(int t, double sample, Map<Integer, String> featuresMap, List<Integer> featuresList) throws Exception {

        int size = featuresList.size();
        double valids = 0, invalids = 0;
        double total = getBinomCoeff(size, t);
        System.out.println(total + " max total " + t + "-sets");


        for (int j = 0; j < sample; j++) {
            TSet set = getITSet(size, t, Math.floor(Math.random() * total), featuresList);
            if (isValidPair(set, featuresMap, featuresList)) {
                valids++;
            } else {
                invalids++;
            }
        }


        valids = 100 * valids / sample;
        invalids = 100 * invalids / sample;

        System.out.println(valids + "% valids, " + invalids + "% invalids");
        System.out.println((valids / 100.0 * total) + " valids, " + (invalids / 100.0 * total) + " invalids");

    }

    public final void parseArgs(String[] args) {

//        //findCoreFeatures(args[0], true);
//        //allPossiblePairs(args[0], true);
//        computeProductPairs(args[0], true, args[1]);
//        //isDimacsValid(args[0], true, args[1]);
//        System.exit(0);
//        SPL.nCk(4, 2);
//        System.exit(0);

//        long t = System.currentTimeMillis();
//        for (int i = 0; i < 15000; i++) {
//            System.out.println(org.apache.commons.math3.util.ArithmeticUtils.binomialCoefficientDouble(15000, 6));
//        }
//        System.out.println(System.currentTimeMillis() - t);


//        for (int i = 0; i <1000000000; i++) {
//        
//        double d = Math.random() * getBinomCoeff(13776, 6);
//
//
//        //System.out.println(getITSet(13776, 6, d));
//        }
//
//        System.exit(0);
        try {
            parser = new CommandLineParser(args, "SPL");
            if (args.length == 0) {
                throw new ParameterException("No arguments");
            }


            parser.parseArgs();
            String command = parser.getCommandName();
            if (command.equals(CommandLineParser.PRIORITIZATION_SOLVER_PRODUCTS)) {
                computePrioritizationSolverProducts(parser.getCommandPrioritizationSolverProducts().fmFile,
                        parser.getCommandPrioritizationSolverProducts().outputFile,
                        parser.getCommandPrioritizationSolverProducts().runs,
                        parser.getCommandPrioritizationSolverProducts().validPairsFile,
                        parser.getCommandPrioritizationSolverProducts().predictable);
            } else if (command.equals(CommandLineParser.AVERAGE_NORMALIZED_DATA_FILES)) {
                if (parser.getCommandAverageDataFiles().outputDirectory == null) {
                    computeAverageDataFiles(parser.getCommandAverageDataFiles().inputDirectory,
                            parser.getCommandAverageDataFiles().inputDirectory,
                            parser.getCommandAverageDataFiles().noNorm);
                } else {
                    computeAverageDataFiles(parser.getCommandAverageDataFiles().inputDirectory,
                            parser.getCommandAverageDataFiles().outputDirectory,
                            parser.getCommandAverageDataFiles().noNorm);
                }
            } else if (command.equals(CommandLineParser.PRIORITIZATION_SPLCAT_PRODUCTS)) {
                computePrioritizationSPLCATProducts(parser.getCommandPrioritizationSPLCATProducts().csvFile,
                        parser.getCommandPrioritizationSPLCATProducts().fmFile,
                        parser.getCommandPrioritizationSPLCATProducts().outputFile,
                        parser.getCommandPrioritizationSPLCATProducts().runs,
                        parser.getCommandPrioritizationSPLCATProducts().validPairsFile);
            } else if (command.equals(CommandLineParser.PRIORITIZATION_SPLCAT_SOLVER_PRODUCTS)) {
                computePrioritizationSPLCATSolverProducts(parser.getCommandPrioritizationSPLCATAndSolverProducts().csvFile,
                        parser.getCommandPrioritizationSPLCATAndSolverProducts().fmFile,
                        parser.getCommandPrioritizationSPLCATAndSolverProducts().outputFile,
                        parser.getCommandPrioritizationSPLCATAndSolverProducts().runs,
                        parser.getCommandPrioritizationSPLCATAndSolverProducts().validPairsFile,
                        parser.commandPrioritizationSPLCATSolverProducts.predictable);
            } else if (command.equals(CommandLineParser.GENERATE_GA)) {
                generateProductsWithGA(parser.getCommandGenerateGA().fmFile,
                        parser.getCommandGenerateGA().csvFile,
                        parser.getCommandGenerateGA().outputFile,
                        parser.getCommandGenerateGA().nbProds,
                        //                        parser.getCommandGenerateGA().popSize,
                        parser.getCommandGenerateGA().runs,
                        parser.getCommandGenerateGA().timeAllowed,
                        parser.getCommandGenerateGA().validPairsFile,
                        parser.getCommandGenerateGA().dimacs,
                        parser.getCommandGenerateGA().noCoverage,
                        parser.getCommandGenerateGA().onlyGA);
            } else if (command.equals(CommandLineParser.NORMALIZE_DATA_FILES)) {
                normalizeDataFile(parser.getCommandNormalizeDataFiles().input);
            } else if (command.equals(CommandLineParser.COMPUTE_PAIRS)) {
                computeValidPairsToFile(parser.getCommandComputePairs().fmFile,
                        parser.getCommandComputePairs().dimacs,
                        parser.getCommandComputePairs().nbParts,
                        parser.getCommandComputePairs().part);
            } else if (command.endsWith(CommandLineParser.COMPUTE_STATS)) {
                computeStats(parser.getCommandComputeStats().fmFile,
                        parser.getCommandComputeStats().dimacs);
            } else if (command.endsWith(CommandLineParser.PRIORITIZATION_PRODUCTS)) {
                computePrioritizationProducts(
                        parser.getCommandPrioritizationProducts().fmFile,
                        parser.getCommandPrioritizationProducts().outputFile,
                        parser.getCommandPrioritizationProducts().runs,
                        parser.getCommandPrioritizationProducts().dimacs,
                        parser.getCommandPrioritizationProducts().nbPairs,
                        parser.getCommandPrioritizationProducts().t,
                        parser.getCommandPrioritizationProducts().prods,
                        parser.getCommandPrioritizationProducts().pairs);
            }
        } catch (ParameterException ex) {
            System.out.println("ERROR: " + ex.getMessage());
            parser.printUsage();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void nCk(int n, int k, Set<TSet> tsets, Map<Integer, String> featuresMap, List<Integer> featuresList) throws Exception {
        int[] a = new int[k];
        nCkH(n, k, 0, a, k, tsets, featuresMap, featuresList);
    }

    public void nCkH(int n, int loopno, int ini, int[] a, int k, Set<TSet> tsets, Map<Integer, String> featuresMap, List<Integer> featuresList) throws Exception {

        if (k == 0) {
            return;
        }

        int i;
        loopno--;
        if (loopno < 0) {
            a[k - 1] = ini - 1;
            TSet p = new TSet();
            for (i = 0; i < k; i++) {
                p.add(featuresList.get(a[i]));
            }
            if (isValidPair(p, featuresMap, featuresList)) {
                tsets.add(p);
            }
            return;

        }
        for (i = ini; i <= n - loopno - 1; i++) {
            a[k - 1 - loopno] = i;
            nCkH(n, loopno, i + 1, a, k, tsets, featuresMap, featuresList);
        }


    }

    public void computePrioritizationProducts(String fmFile, String outputDir, int runs, boolean dimacs, int nbPairs, int t, String prods, String tsets) throws Exception {









        //prio     

//        File output = new File(outputDir);
//        if (!output.isDirectory()) {
//            throw new ParameterException("Output directory specified is incorrect");
//        }
//        if (runs < 1) {
//            throw new ParameterException("Number of runs specified incorrect");
//        }
//
//        if (!outputDir.endsWith("/")) {
//            outputDir += "/";
//        }
//
//        if (!new File(fmFile).exists()) {
//            throw new ParameterException("The specified FM file does not exist");
//        }
//
//
//        this.predictable = false;
//        this.dimacs = dimacs;
//        this.dimacsFile = fmFile;
//        if (!dimacs) {
//            fm = loadFeatureModel(fmFile);
//            fm.loadModel();
//            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
//            reasonerSAT.init();
//        } else {
//            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
//            dimacsSolver.setTimeout(SATtimeout);
//
//            DimacsReader dr = new DimacsReader(dimacsSolver);
//            dr.parseInstance(new FileReader(fmFile));
//        }
//
//        double s = 0;
//
////        double prev = 5000;
////        for (int i = 0; i <= 50; i++) {
////            
////            s += ArithmeticUtils.binomialCoefficientDouble(5000-(int) (i*prev), 6);
////            prev= i*5000.0/100.0;
////        }
////        System.out.println(s/ArithmeticUtils.binomialCoefficientDouble(10000, 6));
//
//        List<Integer> featuresList = new ArrayList<Integer>();
//        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
//        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
//
//        if (!dimacs) {
//            computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
//        } else {
//            computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
//        }
//
//
////        for (int i = 2; i <= 6; i++) {
////
////            for (int j = 0; j < runs; j++) {
////                System.out.println("t = " + i+", run = "+ (j+1));
////                Set<TSet> validTSets = computeNRandValidTSets(featuresMap, featuresList, nbPairs, i);
////                serializeTSets(outputDir + new File(fmFile).getName() + "." + i + "wise.set"+(j+1), validTSets);
////            }
////        }
////        System.exit(0);
////
////        List<Product> products = loadSerializedProducts(prods);
////
////
////        System.out.println("Starting the runs...");
////        for (int i = 0; i < runs; i++) {
////            System.out.println("run " + (i + 1));
////
////            System.out.println("Sampling " + nbPairs + " valid " + t + "-sets...");
////            Set<TSet> validTSets = computeNRandValidTSets(featuresMap, featuresList, nbPairs, t);
////            System.out.println("done, coverage...");
//
////        File[] tsetss = new File(tsets).listFiles();
//
////        for (File ts : tsetss) {
////            System.out.println("handling " + ts);
////            Set<TSet> validTSets = loadSerializedTSets(ts.toString());
//        File[] prodfs = new File(prods).listFiles();
//        
//        for (File f : prodfs) {
//
//            
//            if (f.toString().contains("SimpleGA")) {
//                System.out.println("Non rand...");
//                System.out.println("\thandling " + f);
//                List<Product> products = loadSerializedProducts(f.toString());
//
//                for (int i = 2; i <= 6; i++) {
//                    System.out.println("t="+i);
//                    for (int j = 1; j <= runs; j++) {
//
//                        Set<TSet> validTSets = loadSerializedTSets(tsets + "/" + f.getName().replaceAll("3wise-", "") + "." + i + "wise.pset" + j);
//                        double[] coverage = null;
//                        double[] fitness = null;
//                        int size = products.size();
//                        computeProductsPartialCoverage(products, validTSets);
//                        coverage = new double[size];
//                        for (int k = 0; k < size; k++) {
//                            double cov = products.get(k).getCoverage();
//                            coverage[k] = cov;
//                        }
//                        fitness = computeFitnessSums(products, SimilarityTechnique.JACCARD_DISTANCE);
//                        //run values
//                        String name = f.getName();
//                        writeCoverageAndFitnessValuesToFile(outputDir + name.substring(0, name.length() - 3).replaceAll("3wise-", "") + i + "wise" + ".coverage" + j, coverage, fitness);
//                    }
//                }
//
//                System.out.println("rands...");
//                for (int l = 1; l <= runs; l++) {
//                    
//                    System.out.println("rand run="+l);
//                    shuffle(products);
//                    for (int i = 2; i <= 6; i++) {
//                        System.out.println("t="+i);
//                        for (int j = 1; j <= runs; j++) {
//
//                            Set<TSet> validTSets = loadSerializedTSets(tsets + "/" + f.getName().replaceAll("3wise-", "") + "." + i + "wise.pset" + j);
//                            double[] coverage = null;
//                            double[] fitness = null;
//                            int size = products.size();
//                            computeProductsPartialCoverage(products, validTSets);
//                            coverage = new double[size];
//                            for (int k = 0; k < size; k++) {
//                                double cov = products.get(k).getCoverage();
//                                coverage[k] = cov;
//                            }
//                            fitness = computeFitnessSums(products, SimilarityTechnique.JACCARD_DISTANCE);
//                            //run values
//                            String name = f.getName();
//                            writeCoverageAndFitnessValuesToFile(outputDir + name.substring(0, name.length() - 3).replaceAll("3wise-", "") +i + "wise."+"rand"+l+".coverage" + j, coverage, fitness);
//                        }
//                    }
//
//                }
//
//            }
////        }
//            }
//
//
//
//
//





        //generate 500/1000  

//                File output = new File(outputDir);
//        if (!output.isDirectory()) {
//            throw new ParameterException("Output directory specified is incorrect");
//        }
//        if (runs < 1) {
//            throw new ParameterException("Number of runs specified incorrect");
//        }
//
//        if (!outputDir.endsWith("/")) {
//            outputDir += "/";
//        }
//
//        if (!new File(fmFile).exists()) {
//            throw new ParameterException("The specified FM file does not exist");
//        }
//
//
//        this.predictable = false;
//        this.dimacs = dimacs;
//        this.dimacsFile = fmFile;
//        if (!dimacs) {
//            fm = loadFeatureModel(fmFile);
//            fm.loadModel();
//            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
//            reasonerSAT.init();
//        } else {
//            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
//            dimacsSolver.setTimeout(SATtimeout);
//
//            DimacsReader dr = new DimacsReader(dimacsSolver);
//            dr.parseInstance(new FileReader(fmFile));
//        }
//
//        double s = 0;
//
////        double prev = 5000;
////        for (int i = 0; i <= 50; i++) {
////            
////            s += ArithmeticUtils.binomialCoefficientDouble(5000-(int) (i*prev), 6);
////            prev= i*5000.0/100.0;
////        }
////        System.out.println(s/ArithmeticUtils.binomialCoefficientDouble(10000, 6));
//
//        List<Integer> featuresList = new ArrayList<Integer>();
//        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
//        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
//
//        if (!dimacs) {
//            computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
//        } else {
//            computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
//        }
//
//
////        for (int i = 2; i <= 6; i++) {
////
////            for (int j = 0; j < runs; j++) {
////                System.out.println("t = " + i+", run = "+ (j+1));
////                Set<TSet> validTSets = computeNRandValidTSets(featuresMap, featuresList, nbPairs, i);
////                serializeTSets(outputDir + new File(fmFile).getName() + "." + i + "wise.set"+(j+1), validTSets);
////            }
////        }
////        System.exit(0);
////
////        List<Product> products = loadSerializedProducts(prods);
////
////
////        System.out.println("Starting the runs...");
////        for (int i = 0; i < runs; i++) {
////            System.out.println("run " + (i + 1));
////
////            System.out.println("Sampling " + nbPairs + " valid " + t + "-sets...");
////            Set<TSet> validTSets = computeNRandValidTSets(featuresMap, featuresList, nbPairs, t);
////            System.out.println("done, coverage...");
//
////        File[] tsetss = new File(tsets).listFiles();
//
////        for (File ts : tsetss) {
////            System.out.println("handling " + ts);
////            Set<TSet> validTSets = loadSerializedTSets(ts.toString());
//
//        
//        if (!dimacs) {
//            reasonerSAT.init();
//            ((Solver) reasonerSAT.getSolver()).setOrder(order);
//            solverIterator = new ModelIterator(reasonerSAT.getSolver());
//            solverIterator.setTimeoutMs(iteratorTimeout);
//        } else {
//            ((Solver) dimacsSolver).setOrder(order);
//            solverIterator = new ModelIterator(dimacsSolver);
//            solverIterator.setTimeoutMs(iteratorTimeout);
//        }
////        GA ga = new GA(0000);
//        
//        int n = 2000;
//            List<Product> products = getUnpredictableProducts(n);
//             List<Product> pproducts = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH).prioritize(products);
//
//           System.out.println("done");
////
//
//
//                double[] coverage = null;
//                double[] fitness = null;
//                int size = pproducts.size();
//                Set<TSet> validTSets = loadSerializedTSets(tsets);
//                computeProductsPartialCoverage(pproducts, validTSets);
//                coverage = new double[size];
//                for (int j = 0; j < size; j++) {
//                    double cov = pproducts.get(j).getCoverage();
//                    coverage[j] = cov;
//                }
//                fitness = computeFitnessSums(pproducts, SimilarityTechnique.JACCARD_DISTANCE);
//
//
//
//                //run values
//                
//                writeCoverageAndFitnessValuesToFile(outputDir + new File(fmFile).getName() + "_GA-SimpleGAProducts-" + n + "prods-" + 60000 + "ms-run" + (0 + 1) + ".dat", coverage, fitness);
//            
//        
////        }
//        //}
//        


















////Gen
//        File output = new File(outputDir);
//        if (!output.isDirectory()) {
//            throw new ParameterException("Output directory specified is incorrect");
//        }
//        if (runs < 1) {
//            throw new ParameterException("Number of runs specified incorrect");
//        }
//
//        if (!outputDir.endsWith("/")) {
//            outputDir += "/";
//        }
//
//        if (!new File(fmFile).exists()) {
//            throw new ParameterException("The specified FM file does not exist");
//        }
////
////
//        this.predictable = false;
//        this.dimacs = dimacs;
//        this.dimacsFile = fmFile;
//        if (!dimacs) {
//            fm = loadFeatureModel(fmFile);
//            fm.loadModel();
//            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
//            reasonerSAT.init();
//        } else {
//            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
//            dimacsSolver.setTimeout(SATtimeout);
//
//            DimacsReader dr = new DimacsReader(dimacsSolver);
//            dr.parseInstance(new FileReader(fmFile));
//        }
//
////        double s = 0;
////
////        double prev = 5000;
////        for (int i = 0; i <= 50; i++) {
////            
////            s += ArithmeticUtils.binomialCoefficientDouble(5000-(int) (i*prev), 6);
////            prev= i*5000.0/100.0;
////        }
////        System.out.println(s/ArithmeticUtils.binomialCoefficientDouble(10000, 6));
//
//        List<Integer> featuresList = new ArrayList<Integer>();
//        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
//        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
//
//        if (!dimacs) {
//            computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
//        } else {
//            computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
//        }
//
//        if (!dimacs) {
//            reasonerSAT.init();
//            ((Solver) reasonerSAT.getSolver()).setOrder(order);
//            solverIterator = new ModelIterator(reasonerSAT.getSolver());
//            solverIterator.setTimeoutMs(iteratorTimeout);
//        } else {
//            ((Solver) dimacsSolver).setOrder(order);
//            solverIterator = new ModelIterator(dimacsSolver);
//            solverIterator.setTimeoutMs(iteratorTimeout);
//        }
//
//
////        for (int i = 2; i <= 6; i++) {
////
////            for (int j = 0; j < runs; j++) {
////                System.out.println("t = " + i+", run = "+ (j+1));
////                Set<TSet> validTSets = computeNRandValidTSets(featuresMap, featuresList, nbPairs, i);
////                serializeTSets(outputDir + new File(fmFile).getName() + "." + i + "wise.set"+(j+1), validTSets);
////            }
////        }
////        System.exit(0);
////
////        List<Product> products = loadSerializedProducts(prods);
////
////
////        System.out.println("Starting the runs...");
////        for (int i = 0; i < runs; i++) {
////            System.out.println("run " + (i + 1));
////
////            System.out.println("Sampling " + nbPairs + " valid " + t + "-sets...");
////            Set<TSet> validTSets = computeNRandValidTSets(featuresMap, featuresList, nbPairs, t);
////            System.out.println("done, coverage...");
//
//        //int[] types = new int[]{100, 500, 1000};
//
////        for (int j = 0; j< 10; j++) {
////      
////            
////        
////        List<Product> pps = getUnpredictableProducts(10000);
////        List<Product> pps2 = getUnpredictableProducts(100);
////        
////        Random r = new Random();
////        List<Product> pps3 = new ArrayList<Product>();
////        while (pps3.size() < 100){
////            int i = r.nextInt(pps.size());
////            pps3.add(pps.remove(i));
////        }
////        System.out.println(SimilarityTechnique.getJaccardFitnessSum(pps3) > SimilarityTechnique.getJaccardFitnessSum(pps2));
////        
////        }
////        System.exit(0);
//        String fmn = new File(fmFile).getName();
//
//        File[] fprods = new File(prods).listFiles();
//        for (File fp : fprods) {
//
//            System.out.println("handling "+fp);
////        for (int i = 0; i < types.length; i++) {
////            int count = types[i];
//            List<Product> pproducts = loadSerializedProducts(fp.toString());
////            List<Product> pproducts = getUnpredictableProducts(count);
//
////            List<Product> pproducts = new ArrayList<Product>();
////            pproducts.addAll(loadSerializedProducts("prods/"+fmn+"_GA-UnpredictableProductsPrioritized-"+count+"prods.productspred.ser"));
////            pproducts.addAll(loadSerializedProducts("prods/"+fmn+"_GA-UnpredictableProductsPrioritized-"+count+"prods.productsunpred.ser"));
//
//            //System.out.println(pproducts.size());
//            //List<Product> products = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH).prioritize(pproducts);
//            List<Product> products = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.GREEDY_SEARCH).prioritize(pproducts);
//            //serializeProducts(outputDir + new File(fmFile).getName() + "_GA-UnpredictableProductsPrioritized-" + count + "prods" + ".product.ser", products);
//
//
//            File[] ts = new File(tsets).listFiles();
//
//            for (File tsf : ts) {
//                System.out.println("\t"+tsf);
////                for (int tt = 2; tt <= 6; tt++) {
////                    System.out.println("t=" + tt);
//                    Set<TSet> validTSets = loadSerializedTSets(tsf.toString());
//                    //computeNRandValidTSets(featuresMap, featuresList, nbPairs * 10, tt);
//                    //serializeTSets(outputDir + new File(fmFile).getName() + "." + tt + "wise.set", validTSets);
//
//
//
//                    //File[] tsetss = new File(tsets).listFiles();
//
//
//                    //Set<TSet> validTSets = loadSerializedTSets(ts.toString());
////                File[] prodfs = new File(prods).listFiles();
////                for (File f : prodfs) {
////                    System.out.println("\thandling " + f);
////                    List<Product> products = loadSerializedProducts(f.toString());
//
////            for (int i = 2; i <= 6; i++) {
////
////            for (int j = 0; j < runs; j++) {
////                System.out.println("t = " + i+", run = "+ (j+1));
////                Set<TSet> validTSets = computeNRandValidTSets(products, nbPairs, i);
////                //Set<TSet> validTSets = computeNRandValidTSets(featuresMap, featuresList, nbPairs, i);
////                serializeTSets(outputDir + f.getName().replaceAll("3wise-", "") + "." + i + "wise.pset"+(j+1), validTSets);
////            }
////        }
//
//
////            }
//
//                    double[] coverage = null;
//                    double[] fitness = null;
//                    int size = products.size();
//                    computeProductsPartialCoverage(products, validTSets);
//                    coverage = new double[size];
//                    for (int j = 0; j < size; j++) {
//                        double cov = products.get(j).getCoverage();
//                        coverage[j] = cov;
//                    }
//                    fitness = computeFitnessSums(products, SimilarityTechnique.JACCARD_DISTANCE);
//
//
//                    String ext;
////                if (dimacs) {
////                    ext = ts.toString().substring(ts.toString().indexOf(".dimacs") + 8, ts.toString().length());
////                } else {
////                    ext = ts.toString().substring(ts.toString().indexOf(".xml") + 5, ts.toString().length());
////                }
//
//                    //run values
//
//                    writeCoverageAndFitnessValuesToFile(outputDir + fp.getName() + ".greedy." + tsf.getName() + ".coverage", coverage, fitness);
////                writeCoverageAndFitnessValuesToFile(outputDir + new File(fmFile).getName() + "_GA-UnpredictableProductsPrioritized-" + count + "prods" + ".products.ser." + tt + "wise." + "coverage", coverage, fitness);
////writeCoverageAndFitnessValuesToFile(outputDir + new File(fmFile).getName() + "_GA-UnpredictableProductsPrioritized-" + count + "prods" + ".products.ser."+tt + "wise."+"rand"+l+".coverage" + j, coverage, fitness);
//
////        }
//
//                    for (int j = 1; j <= 10; j++) {
//
//                        System.out.println("rand "+j);
//                        shuffle(pproducts);
//                        double[] coverager = null;
//                        double[] fitnessr = null;
//                        computeProductsPartialCoverage(pproducts, validTSets);
//                        coverager = new double[size];
//                        for (int k = 0; k < size; k++) {
//                            double cov = pproducts.get(k).getCoverage();
//                            coverager[k] = cov;
//                        }
//                        fitnessr = computeFitnessSums(pproducts, SimilarityTechnique.JACCARD_DISTANCE);
//
////                if (dimacs) {
////                    ext = ts.toString().substring(ts.toString().indexOf(".dimacs") + 8, ts.toString().length());
////                } else {
////                    ext = ts.toString().substring(ts.toString().indexOf(".xml") + 5, ts.toString().length());
////                }
//
//                        //run values
//
//                        writeCoverageAndFitnessValuesToFile(outputDir + fp.getName() + ".greedy." + tsf.getName() + ".rand" + j + ".coverage", coverager, fitnessr);
////                    writeCoverageAndFitnessValuesToFile(outputDir + new File(fmFile).getName() + "_GA-UnpredictableProductsPrioritized-" + count + "prods" + ".products.ser." + tt + "wise." + "rand" + j + ".coverage", coverager, fitnessr);
//
//                    }
//                }
////            }
//            // }
//        }













//
//        //last gen...
//
        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }

        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }
//
//
        this.predictable = false;
        this.dimacs = dimacs;
        this.dimacsFile = fmFile;
        if (!dimacs) {
            fm = loadFeatureModel(fmFile);
            fm.loadModel();
            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
            reasonerSAT.init();
        } else {
            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
            dimacsSolver.setTimeout(SATtimeout);

            DimacsReader dr = new DimacsReader(dimacsSolver);
            dr.parseInstance(new FileReader(fmFile));
        }


        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();

        if (!dimacs) {
            computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
        } else {
            computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
        }

        if (!dimacs) {
            reasonerSAT.init();
            ((Solver) reasonerSAT.getSolver()).setOrder(order);
            solverIterator = new ModelIterator(reasonerSAT.getSolver());
            solverIterator.setTimeoutMs(iteratorTimeout);
        } else {
            ((Solver) dimacsSolver).setOrder(order);
            solverIterator = new ModelIterator(dimacsSolver);
            solverIterator.setTimeoutMs(iteratorTimeout);
        }



        List<Product> eval = loadSerializedProducts(prods);
        Set<TSet> ts = loadSerializedTSets(tsets);

        String evalN = new File(prods).getName();
        String tsN = new File(tsets).getName();
        double[] coverage = null;
        double[] fitness = null;
        int size = eval.size();
        computeProductsPartialCoverage(eval, ts);
        coverage = new double[size];
        for (int j = 0; j < size; j++) {
            double cov = eval.get(j).getCoverage();
            coverage[j] = cov;
        }
        fitness = computeFitnessSums(eval, SimilarityTechnique.JACCARD_DISTANCE);

        writeCoverageAndFitnessValuesToFile(outputDir + evalN+"."+tsN+ ".coverage", coverage, fitness);


        System.exit(0);


//        for (int j = 0; j< 10; j++) {
//      
//            
//        
//        List<Product> pps = getUnpredictableProducts(10000);
//        List<Product> pps2 = getUnpredictableProducts(100);
//        
//        Random r = new Random();
//        List<Product> pps3 = new ArrayList<Product>();
//        while (pps3.size() < 100){
//            int i = r.nextInt(pps.size());
//            pps3.add(pps.remove(i));
//        }
//        System.out.println(SimilarityTechnique.getJaccardFitnessSum(pps3) > SimilarityTechnique.getJaccardFitnessSum(pps2));
//        
//        }
//        System.exit(0);

        int count = 2000;

        List<Product> products = getUnpredictableProducts(count);
        long last = System.currentTimeMillis();
        System.out.println("serializing..");
        serializeProducts(outputDir + new File(fmFile).getName() + "_GA-UnpredictableProducts-" + count + "prods" + ".product.ser", products);
        System.out.println("done in " + (System.currentTimeMillis() - last));
        last = System.currentTimeMillis();
//        System.out.println("Prioritizing " + count + " prods..."+ last);
//        
//        List<Product> pproducts = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH).prioritize(products);
//        System.out.println("done in " + (System.currentTimeMillis() - last));
//        last = System.currentTimeMillis();
//        System.out.println("serializing..");
//        serializeProducts(outputDir + new File(fmFile).getName() + "_GA-UnpredictableProductsPrioritized-" + count + "prods" + ".product.ser", pproducts);
//        System.out.println("done in " + (System.currentTimeMillis() - last));

        //Set<TSet> validTSets = loadSerializedTSets(tsets);


//        double[] coverage = null;
//        double[] fitness = null;
//        int size = products.size();
//        computeProductsPartialCoverage(products, validTSets);
//        coverage = new double[size];
//        for (int j = 0; j < size; j++) {
//            double cov = products.get(j).getCoverage();
//            coverage[j] = cov;
//        }
//        fitness = computeFitnessSums(products, SimilarityTechnique.JACCARD_DISTANCE);
//
//        writeCoverageAndFitnessValuesToFile(outputDir + new File(fmFile).getName() + "_GA-UnpredictableProducts-" + count + "prods" + ".products.ser." + 6 + "wise." + "coverage", coverage, fitness);

        GA ga = new GA(1);
        ga.runSimpleGA3(products);
//        double[] coveragep = null;
//        double[] fitnessp = null;
//        size = pproducts.size();
//        computeProductsPartialCoverage(pproducts, validTSets);
//        coveragep = new double[size];
//        for (int j = 0; j < size; j++) {
//            double cov = pproducts.get(j).getCoverage();
//            coveragep[j] = cov;
//        }
//        fitnessp = computeFitnessSums(products, SimilarityTechnique.JACCARD_DISTANCE);
//        writeCoverageAndFitnessValuesToFile(outputDir + new File(fmFile).getName() + "_GA-UnpredictableProductsPrioritized-" + count + "prods" + ".products.ser." + 6 + "wise." + "coverage", coveragep, fitnessp);


//        }



    }

    public void computePrioritizationProducts(String fmFile, String outputDir, int runs, boolean dimacs, int nbPairs, int t, int nbProds, long timeAllowed) throws Exception {


        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }

        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }


        if (nbProds < 0) {
            throw new ParameterException("Negative nbRuns");
        }


        this.predictable = false;
        this.dimacs = dimacs;
        this.dimacsFile = fmFile;
        if (!dimacs) {
            fm = loadFeatureModel(fmFile);
            fm.loadModel();
            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
            reasonerSAT.init();
        } else {
            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
            dimacsSolver.setTimeout(SATtimeout);

            DimacsReader dr = new DimacsReader(dimacsSolver);
            dr.parseInstance(new FileReader(fmFile));
        }




        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();

        if (!dimacs) {
            computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
        } else {
            computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
        }


        System.out.println(featuresList.size() + " features");

//        System.out.println("Sampling " + nbPairs + " valid " + t + "-sets...");
//        Set<TSet> validTSets = computeNRandValidTSets(featuresMap, featuresList, nbPairs, t);

        double[] coverageValuesUnpredictable = new double[nbProds];
        double[] fitnessValuesUnpredictable = new double[nbProds];
        double[] coverageValuesSimpleGA = new double[nbProds];
        double[] fitnessValuesSimpleGA = new double[nbProds];


        if (!dimacs) {
            reasonerSAT.init();
            ((Solver) reasonerSAT.getSolver()).setOrder(order);
            solverIterator = new ModelIterator(reasonerSAT.getSolver());
            solverIterator.setTimeoutMs(iteratorTimeout);
        } else {
            ((Solver) dimacsSolver).setOrder(order);
            solverIterator = new ModelIterator(dimacsSolver);
            solverIterator.setTimeoutMs(iteratorTimeout);
        }

        GA ga = new GA(timeAllowed);
        String sNbProds = "" + nbProds;

        String fmFileName = new File(fmFile).getName();
        System.out.println("Starting the runs...");
        for (int i = 0; i < runs; i++) {
            System.out.println("run " + (i + 1));


            double[] runCoverageUnpredictable = null;
            double[] fitnessUnpredictable = null;
            List<Product> unpredictableProducts = null;
            List<Product> simOptiPrioritizedUnpredictable = null;

            //unpredictable products
            System.out.println("Unpredictable...");
            unpredictableProducts = getUnpredictableProducts(nbProds);
            shuffle(unpredictableProducts);
            System.out.println("done, coverage...");

//            computeProductsPartialCoverage(unpredictableProducts, validTSets);
//            runCoverageUnpredictable = new double[coverageValuesUnpredictable.length];
//
//            for (int j = 0; j < nbProds; j++) {
//                double cov = unpredictableProducts.get(j).getCoverage();
//                coverageValuesUnpredictable[j] += cov;
//                runCoverageUnpredictable[j] = cov;
//            }
//
//            fitnessUnpredictable = computeFitnessSums(unpredictableProducts, SimilarityTechnique.JACCARD_DISTANCE);
//            for (int j = 0; j < fitnessUnpredictable.length; j++) {
//                fitnessValuesUnpredictable[j] += fitnessUnpredictable[j];
//
//            }

            System.out.println("Simple GA...");
            List<Product> gaSimpleRes = ga.runSimpleGA(nbProds, Individual.MUTATE_WORST).getProducts();


            System.out.println("done, coverage...");


//            double[] runCoverageGA = null;
//            double[] fitnessSimpleGA = null;
//
//            computeProductsPartialCoverage(gaSimpleRes, validTSets);
//            runCoverageGA = new double[coverageValuesSimpleGA.length];
//            for (int j = 0; j < nbProds; j++) {
//                double cov = gaSimpleRes.get(j).getCoverage();
//                coverageValuesSimpleGA[j] += cov;
//                runCoverageGA[j] = cov;
//            }
//            fitnessSimpleGA = computeFitnessSums(gaSimpleRes, SimilarityTechnique.JACCARD_DISTANCE);
//            for (int j = 0; j < fitnessValuesSimpleGA.length; j++) {
//                fitnessValuesSimpleGA[j] += fitnessSimpleGA[j];
//
//            }


            //run values

//            writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProducts-" + t + "wise-" + sNbProds + "prods-" + timeAllowed + "ms-run" + (i + 1) + ".dat", runCoverageUnpredictable, fitnessUnpredictable);
//            writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-SimpleGAProducts-" + t + "wise-" + sNbProds + "prods-" + timeAllowed + "ms-run" + (i + 1) + ".dat", runCoverageGA, fitnessSimpleGA);

            //save products
            writeProductsToFile(outputDir + fmFileName + "_GA-UnpredictableProducts-" + t + "wise-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.csv", unpredictableProducts, featuresMap, featuresList);
            writeProductsToFile(outputDir + fmFileName + "_GA-SimpleGAProducts-" + t + "wise-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.csv", gaSimpleRes, featuresMap, featuresList);

            serializeProducts(outputDir + fmFileName + "_GA-UnpredictableProducts-" + t + "wise-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.ser", unpredictableProducts);
            serializeProducts(outputDir + fmFileName + "_GA-SimpleGAProducts-" + t + "wise-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.ser", gaSimpleRes);

        }

//        for (int j = 0; j < nbProds; j++) {
//
//            coverageValuesUnpredictable[j] /= runs;
//
//
//            fitnessValuesUnpredictable[j] /= runs;
//
//
//            coverageValuesSimpleGA[j] /= runs;
//            fitnessValuesSimpleGA[j] /= runs;
//        }
//
//
//
//        writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProducts-" + t + "wise-" + sNbProds + "prods-" + timeAllowed + "ms-" + runs + "runs.dat", coverageValuesUnpredictable, fitnessValuesUnpredictable);
//        writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-SimpleGAProducts-" + t + "wise-" + sNbProds + "prods-" + timeAllowed + "ms-" + runs + "runs.dat", coverageValuesSimpleGA, fitnessValuesSimpleGA);

    }

    public void computePrioritizationSPLCATProducts(String splcatFile, String fmFile, String outputDir, int runs, String validPairsFile) throws Exception {
        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }
        File splcatCSV = new File(splcatFile);
        if (!splcatCSV.exists()) {
            throw new ParameterException("The specified SPLCAT file does not exist");
        }
        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (validPairsFile != null && !new File(validPairsFile).exists()) {
            throw new ParameterException("The specified valid pairs file does not exist");
        }
        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM (xml) file does not exist");
        }

        fm = loadFeatureModel(fmFile);
        fm.loadModel();
        reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
        reasonerSAT.init();

        String[] features = reasonerSAT.getVarIndex2NameMap();
        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
        computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);

        String splcatFileName = splcatCSV.getName();

        List<Product> splcatProducts = loadProductsFromCSVFile(splcatCSV, featuresMapRev);
        int productsSize = splcatProducts.size();

        Set<TSet> validPairs;
        if (validPairsFile != null) {
            validPairs = loadPairs(validPairsFile);
        } else {
            validPairs = computeValidPairs(featuresMap, featuresList, null, false, null, 1, 1);


            //validPairs = computeNRandValidPairs(featuresMap, featuresList, null, false, null, 1, 1, 8);
            //System.out.println(validPairs.size());


            //validPairs = computePairsCoveredByProducts(splcatProducts);
        }


        SimilarityTechnique simJaccardGreedy = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.GREEDY_SEARCH);
        SimilarityTechnique simJaccardOpti = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH);
        RandomTechnique random = new RandomTechnique();

        double[] coverageValuesOriginal = new double[productsSize];
        double[] fitnessValuesOriginal;
        double[] coverageValuesRand = new double[productsSize];
        double[] fitnessValuesRand = new double[productsSize];
        double[] coverageValuesSimGreedy = new double[productsSize];
        double[] fitnessValuesSimGreedy;
        double[] coverageValuesSimOpti = new double[productsSize];
        double[] fitnessValuesSimOpti;

        // Original order
        computeProductsCoverage(splcatProducts, validPairs);
        for (int j = 0; j < productsSize; j++) {
            coverageValuesOriginal[j] += splcatProducts.get(j).getCoverage();
        }

        fitnessValuesOriginal = computeFitnessSums(splcatProducts, SimilarityTechnique.JACCARD_DISTANCE);
        shuffle(splcatProducts);

        // Random
        for (int i = 0; i < runs; i++) {


            List<Product> randomProducts = random.prioritize(splcatProducts);
            computeProductsCoverage(randomProducts, validPairs);
            for (int j = 0; j < randomProducts.size(); j++) {
                coverageValuesRand[j] += randomProducts.get(j).getCoverage();
            }
            double[] fitnessRand = computeFitnessSums(randomProducts, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessRand.length; j++) {
                fitnessValuesRand[j] += fitnessRand[j];

            }
        }

        for (int i = 0; i < coverageValuesRand.length; i++) {
            coverageValuesRand[i] /= runs;
            fitnessValuesRand[i] /= runs;
        }

        // Similarity (Greedy)
        List<Product> simGreedyPrioritized = simJaccardGreedy.prioritize(splcatProducts);
        computeProductsCoverage(simGreedyPrioritized, validPairs);
        for (int j = 0; j < simGreedyPrioritized.size(); j++) {
            coverageValuesSimGreedy[j] += simGreedyPrioritized.get(j).getCoverage();
        }
        fitnessValuesSimGreedy = computeFitnessSums(simGreedyPrioritized, SimilarityTechnique.JACCARD_DISTANCE);

        // Similarity (Near optimal)
        List<Product> simOptiPrioritized = simJaccardOpti.prioritize(splcatProducts);
        computeProductsCoverage(simOptiPrioritized, validPairs);
        for (int j = 0; j < simOptiPrioritized.size(); j++) {
            coverageValuesSimOpti[j] += simOptiPrioritized.get(j).getCoverage();
        }
        fitnessValuesSimOpti = computeFitnessSums(simOptiPrioritized, SimilarityTechnique.JACCARD_DISTANCE);

        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProducts-OriginalOrder.dat", coverageValuesOriginal, fitnessValuesOriginal);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProducts-Random-" + runs + "runs.dat", coverageValuesRand, fitnessValuesRand);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProducts-SimJaccardGreedy.dat", coverageValuesSimGreedy, fitnessValuesSimGreedy);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProducts-SimJaccardOpti.dat", coverageValuesSimOpti, fitnessValuesSimOpti);

    }

    public static Product getJaccardWorstProduct(List<Product> prods) {
        double dMin = 0.0;
        Product worst = prods.get(0);

        for (int i = 1; i < prods.size(); i++) {
            dMin += DistancesUtil.getJaccardDistance(worst, prods.get(i));
        }

        for (int i = 1; i < prods.size(); i++) {
            double dist = 0.0;
            Product p = prods.get(i);
            for (int j = 0; j < prods.size(); j++) {
                dist += DistancesUtil.getJaccardDistance(p, prods.get(j));
            }
            if (dist < dMin) {
                dMin = dist;
                worst = p;
            }

        }
        return worst;
    }

    public void computeStats(String fmFile, boolean dimacs) throws Exception {
        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }

        if (!dimacs) {
            fm = loadFeatureModel(fmFile);
            fm.loadModel();
            new FTPreOrderSortedECTraversalHeuristic("Pre-CL-MinSpan", fm, FTPreOrderSortedECTraversalHeuristic.FORCE_SORT);
            VariableOrderingHeuristic heuristic = VariableOrderingHeuristicsManager.createHeuristicsManager().getHeuristic("Pre-CL-MinSpan");

            // Creates the BDD reasoner
            ReasoningWithBDD reasonerBDD = new FMReasoningWithBDD(fm, heuristic, 50000, 50000, 60000, "pre-order");

            // Initialize the reasoner (BDD is created at this moment)
            reasonerBDD.init();
            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
            reasonerSAT.init();
            System.out.println("#Configs: " + reasonerBDD.countValidConfigurations());
            System.out.println("#Constraints: " + reasonerSAT.getSolver().nConstraints());
            System.out.println("#Features: " + reasonerSAT.getSolver().nVars());

        } else {
            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
            dimacsSolver.setTimeout(SATtimeout);
            DimacsReader dr = new DimacsReader(dimacsSolver);
            dr.parseInstance(new FileReader(fmFile));
            System.out.println("#Constraints: " + dimacsSolver.nConstraints());
            System.out.println("#Features: " + dimacsSolver.nVars());
        }
    }

    public void generateProductsWithGA(String fmFile, String splcatFile, String outputDir, int nbProds, /*int popSize,*/ int runs, long timeAllowed, String validPairsFile, boolean dimacs, boolean noCoverage, boolean onlyGA) throws Exception {



        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }

        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }

        if (validPairsFile != null && !new File(validPairsFile).exists()) {
            throw new ParameterException("The specified valid pairs file does not exist");
        }

        if (nbProds < 0 && splcatFile == null) {
            throw new ParameterException("If -nbRuns < 0 then the csv file is mandatory!");
        }

        File splcatCSV = null;
        if (splcatFile != null) {
            splcatCSV = new File(splcatFile);
            if (!splcatCSV.exists()) {
                throw new ParameterException("The specified SPLCAT file does not exist");
            }
        }

        this.predictable = false;
        this.dimacs = dimacs;
        this.dimacsFile = fmFile;
        if (!dimacs) {
            fm = loadFeatureModel(fmFile);
            fm.loadModel();
            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
            reasonerSAT.init();
        } else {
            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
            dimacsSolver.setTimeout(SATtimeout);

            DimacsReader dr = new DimacsReader(dimacsSolver);
            dr.parseInstance(new FileReader(fmFile));
        }




        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();

        if (!dimacs) {
            computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
        } else {
            computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
        }


        System.out.println(featuresMapRev.size() + " features");

        Set<TSet> validPairs = null;

        if (!noCoverage) {

            if (validPairsFile != null) {
                System.out.println("Loading valid pairs from the file...");
                validPairs = loadPairs(validPairsFile);
            }
        }

        String sNbProds = "" + nbProds;

        if (nbProds < 0) {

            List<Product> splcatProducts = loadProductsFromCSVFile(splcatCSV, featuresMapRev);
            nbProds = splcatProducts.size();
            sNbProds = "SPLCAT";
            if (!noCoverage) {
                if (validPairs == null) {
                    validPairs = computePairsCoveredByProducts(splcatProducts);
                }
            }
        }

        if (!noCoverage) {
            if (validPairs == null) {
                if (!dimacs) {
                    validPairs = computeValidPairs(featuresMap, featuresList, null, false, null, 1, 1);
                } else {
                    computeValidPairs(featuresMap, featuresList, (fmFile + ".validpairs"), true, dimacsSolver, 1, 1);
                }
            }

            System.out.println(validPairs.size() + " valid pairs.");
        }
        System.out.println(nbProds + " products to generate, " + runs + " runs");


        this.estimateValidTSets(6, 1000, featuresMap, featuresList);

        System.exit(0);


        double[] coverageValuesUnpredictable = new double[nbProds];
        double[] fitnessValuesUnpredictable = new double[nbProds];
        double[] coverageValuesUnpredictablePrioritized = new double[nbProds];
        double[] fitnessValuesUnpredictablePrioritized = new double[nbProds];
        double[] coverageValuesSimpleGA = new double[nbProds];
        double[] fitnessValuesSimpleGA = new double[nbProds];


        if (!dimacs) {
            reasonerSAT.init();
            ((Solver) reasonerSAT.getSolver()).setOrder(order);
            solverIterator = new ModelIterator(reasonerSAT.getSolver());
            solverIterator.setTimeoutMs(iteratorTimeout);
        } else {
            ((Solver) dimacsSolver).setOrder(order);
            solverIterator = new ModelIterator(dimacsSolver);
            solverIterator.setTimeoutMs(iteratorTimeout);
        }

        GA ga = new GA(timeAllowed);
        String fmFileName = new File(fmFile).getName();
        System.out.println("Starting the runs...");
        for (int i = 0; i < runs; i++) {
            System.out.println("run " + (i + 1));


            double[] runCoverageUnpredictable = null;
            double[] fitnessUnpredictable = null;
            double[] runCoverageUnpredictablePrioritized = null;
            double[] fitnessUnpredictablePrioritized = null;
            List<Product> unpredictableProducts = null;
            List<Product> simOptiPrioritizedUnpredictable = null;
            if (!onlyGA) {

                //unpredictable products
                System.out.println("Unpredictable...");
                unpredictableProducts = getUnpredictableProducts(nbProds);

                if (!noCoverage) {
                    System.out.println("done, coverage...");
                } else {
                    System.out.println("done.");
                }
                shuffle(unpredictableProducts);



                if (!noCoverage) {

                    computeProductsCoverage(unpredictableProducts, validPairs);
                    runCoverageUnpredictable = new double[coverageValuesUnpredictable.length];

                    for (int j = 0; j < nbProds; j++) {
                        double cov = unpredictableProducts.get(j).getCoverage();
                        coverageValuesUnpredictable[j] += cov;
                        runCoverageUnpredictable[j] = cov;
                    }

                    fitnessUnpredictable = computeFitnessSums(unpredictableProducts, SimilarityTechnique.JACCARD_DISTANCE);
                    for (int j = 0; j < fitnessUnpredictable.length; j++) {
                        fitnessValuesUnpredictable[j] += fitnessUnpredictable[j];

                    }
                }

                //unpredictable prioritized
                System.out.println("unpredictable prioritized...");
                simOptiPrioritizedUnpredictable = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH).prioritize(unpredictableProducts);
                if (!noCoverage) {
                    System.out.println("done, coverage...");
                } else {
                    System.out.println("done.");
                }


                if (!noCoverage) {

                    computeProductsCoverage(simOptiPrioritizedUnpredictable, validPairs);
                    runCoverageUnpredictablePrioritized = new double[coverageValuesUnpredictablePrioritized.length];
                    for (int j = 0; j < nbProds; j++) {
                        double cov = simOptiPrioritizedUnpredictable.get(j).getCoverage();
                        coverageValuesUnpredictablePrioritized[j] += cov;
                        runCoverageUnpredictablePrioritized[j] = cov;
                    }
                    fitnessUnpredictablePrioritized = computeFitnessSums(simOptiPrioritizedUnpredictable, SimilarityTechnique.JACCARD_DISTANCE);
                    for (int j = 0; j < fitnessValuesUnpredictablePrioritized.length; j++) {
                        fitnessValuesUnpredictablePrioritized[j] += fitnessUnpredictablePrioritized[j];

                    }
                }

            }

            System.out.println("Simple GA...");


//            int a = 1;
//            while (a < 5) {
//                Product p = getUnpredictableProduct();
//                //if (p.getSelectedNumber() < 2000)
//                    System.out.println(p.getSelectedNumber());
//                Thread.sleep(100);
//            }

//            System.out.println(isDimacsValid(featuresMap));
//            System.exit(0);


            List<Product> gaSimpleRes = ga.runSimpleGA(nbProds, Individual.MUTATE_WORST).getProducts();
            //List<Product> gaSimpleRes = ga.runGA(nbProds, 5).getProducts();
//            int a = 1;
//
//            List<Product> gaSimpleRes = null;
//            int nbsuc = 0, nbfail = 0;
//            List<Product> smallProducts = new ArrayList<Product>();
//            while (a < 5) {
////                gaSimpleRes = ga.runSimpleGA(nbProds, Individual.MUTATE_WORST).getProducts();
//                //List<Product> gaSimpleRes = ga.runGA(nbProds, 5).getProducts();
//
////                System.out.println("succ: " + nbsuc + " fails:" + nbfail);
//                try {
//
//                    Product p = getUnpredictableProduct();
//                    if (p.getSelectedNumber() < 300) {
//                        System.out.println(p.getSelectedNumber());
//                        if (!smallProducts.contains(p)) {
//                            smallProducts.add(p);
//                            if (smallProducts.size() >= 15){
//                                writeProductsToFile("15prods.csv", smallProducts, featuresMap, featuresList);
//                                System.exit(0);
//                            }
//                        }
//
//                    }
////                    Product p = gaSimpleRes.get(0);
//////
//////                    System.out.println(p.getSelectedNumber());
//////                    System.out.println(p);
////                    IVecInt prod = new VecInt(p.size());
////                    prod.push(1);
////                    for (int j = 2; j <= 800; j++) {
////                        if (Math.random() > 0.8) {
////                            prod.push(j);
////                        } else {
////                            prod.push(-j);
////                        }
////                    }
////                    //reasonerSAT.getSolver().reset();
//////                    toProduct(dimacsSolver.findModel(productToInt(p)));
////                    Product tp = toProduct(dimacsSolver.findModel(prod));
////                    System.out.println(tp.getSelectedNumber() + " "+tp);
////                    nbsuc++;
//                } catch (Exception e) {
//                    nbfail++;
//                }
//                Thread.sleep(100);
//            }
//
//
//            System.exit(0);

            if (!noCoverage) {
                System.out.println("done, coverage...");
            } else {
                System.out.println("done.");
            }


            double[] runCoverageGA = null;
            double[] fitnessSimpleGA = null;
            if (!noCoverage) {
                computeProductsCoverage(gaSimpleRes, validPairs);
                runCoverageGA = new double[coverageValuesSimpleGA.length];
                for (int j = 0; j < nbProds; j++) {
                    double cov = gaSimpleRes.get(j).getCoverage();
                    coverageValuesSimpleGA[j] += cov;
                    runCoverageGA[j] = cov;
                }
                fitnessSimpleGA = computeFitnessSums(gaSimpleRes, SimilarityTechnique.JACCARD_DISTANCE);
                for (int j = 0; j < fitnessValuesSimpleGA.length; j++) {
                    fitnessValuesSimpleGA[j] += fitnessSimpleGA[j];

                }
            }

            //run values
            if (!noCoverage) {
                if (!onlyGA) {
                    writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProducts-" + sNbProds + "prods-" + timeAllowed + "ms-run" + (i + 1) + ".dat", runCoverageUnpredictable, fitnessUnpredictable);
                    writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProductsPrioritized-" + sNbProds + "prods-" + timeAllowed + "ms-run" + (i + 1) + ".dat", runCoverageUnpredictablePrioritized, fitnessUnpredictablePrioritized);
                }
                writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-SimpleGAProducts-" + sNbProds + "prods-" + timeAllowed + "ms-run" + (i + 1) + ".dat", runCoverageGA, fitnessSimpleGA);
            }
            //save products

            if (!onlyGA) {
                writeProductsToFile(outputDir + fmFileName + "_GA-UnpredictableProducts-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.csv", unpredictableProducts, featuresMap, featuresList);
                writeProductsToFile(outputDir + fmFileName + "_GA-UnpredictablePrioritized-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.csv", simOptiPrioritizedUnpredictable, featuresMap, featuresList);
            }
            writeProductsToFile(outputDir + fmFileName + "_GA-SimpleGAProducts-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.csv", gaSimpleRes, featuresMap, featuresList);
        }

        if (!noCoverage) {
            for (int i = 0; i < nbProds; i++) {
                if (!onlyGA) {
                    coverageValuesUnpredictable[i] /= runs;
                    coverageValuesUnpredictablePrioritized[i] /= runs;

                    fitnessValuesUnpredictable[i] /= runs;
                    fitnessValuesUnpredictablePrioritized[i] /= runs;
                }
                coverageValuesSimpleGA[i] /= runs;
                fitnessValuesSimpleGA[i] /= runs;
            }
        }

        if (!noCoverage) {
            if (!onlyGA) {
                writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProducts-" + sNbProds + "prods-" + timeAllowed + "ms-" + runs + "runs.dat", coverageValuesUnpredictable, fitnessValuesUnpredictable);
            }
            writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProductsPrioritized-" + sNbProds + "prods-" + timeAllowed + "ms-" + runs + "runs.dat", coverageValuesUnpredictablePrioritized, fitnessValuesUnpredictablePrioritized);
            writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-SimpleGAProducts-" + sNbProds + "prods-" + timeAllowed + "ms-" + runs + "runs.dat", coverageValuesSimpleGA, fitnessValuesSimpleGA);
        }
    }

//    public void allPossiblePairs(String fmFile, boolean dimacs) {
//        try {
//            if (!new File(fmFile).exists()) {
//                throw new ParameterException("The specified FM file does not exist");
//            }
//
//            this.predictable = false;
//            this.dimacs = dimacs;
//            this.dimacsFile = fmFile;
//            if (!dimacs) {
//                fm = loadFeatureModel(fmFile);
//                fm.loadModel();
//                reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
//                reasonerSAT.init();
//            } else {
//                dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
//                dimacsSolver.setTimeout(SATtimeout);
//                DimacsReader dr = new DimacsReader(dimacsSolver);
//                dr.parseInstance(new FileReader(fmFile));
//            }
//
//            List<Integer> featuresList = new ArrayList<Integer>();
//            Map<Integer, String> featuresMap = new HashMap<Integer, String>();
//            Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
//
//            if (!dimacs) {
//                computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
//            } else {
//                computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
//            }
//
//
//
//            List<FeaturesPair> pairs = new ArrayList<FeaturesPair>();
//
//            for (int i = 0; i < featuresList.size(); i++) {
//                for (int j = 0; j < featuresList.size(); j++) {
//                    if (j > i) {
//                        int left = featuresList.get(i);
//                        int right = featuresList.get(j);
//                        if (Math.abs(left) != Math.abs(right)) {
//                            FeaturesPair pair = new FeaturesPair(left, right);
//                            //pairs.add(pair);
//                            System.err.println(pair.getX() + ";" + pair.getY());
//                        }
//                    }
//
//                }
//                System.out.println(i);
//
//
//
//            }
//
//
//        } catch (Exception ex) {
//            Logger.getLogger(SPL.class.getName()).log(Level.SEVERE, null, ex);
//            System.out.println(
//                    "error");
//        }
//
//    }
    public void findCoreFeatures(String fmFile, boolean dimacs) {
        try {
            if (!new File(fmFile).exists()) {
                throw new ParameterException("The specified FM file does not exist");
            }

            this.predictable = false;
            this.dimacs = dimacs;
            this.dimacsFile = fmFile;
            if (!dimacs) {
                fm = loadFeatureModel(fmFile);
                fm.loadModel();
                reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
                reasonerSAT.init();
            } else {
                dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                dimacsSolver.setTimeout(SATtimeout);
                DimacsReader dr = new DimacsReader(dimacsSolver);
                dr.parseInstance(new FileReader(fmFile));
            }

            List<Integer> featuresList = new ArrayList<Integer>();
            Map<Integer, String> featuresMap = new HashMap<Integer, String>();
            Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();

            if (!dimacs) {
                computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
            } else {
                computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
            }


            if (!dimacs) {
                reasonerSAT.init();
                ((Solver) reasonerSAT.getSolver()).setOrder(order);
                solverIterator = new ModelIterator(reasonerSAT.getSolver());
                solverIterator.setTimeoutMs(iteratorTimeout);
            } else {
                ((Solver) dimacsSolver).setOrder(order);
                solverIterator = new ModelIterator(dimacsSolver);
                solverIterator.setTimeoutMs(iteratorTimeout);
            }


            int a = 1;

            while (a < 5) {

                Product p = getUnpredictableProduct();
                IVecInt t = new VecInt();

                this.predictable = false;
                this.dimacs = dimacs;
                this.dimacsFile = fmFile;
                if (!dimacs) {
                    fm = loadFeatureModel(fmFile);
                    fm.loadModel();
                    reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
                    reasonerSAT.init();
                } else {
                    dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                    dimacsSolver.setTimeout(SATtimeout);
                    DimacsReader dr = new DimacsReader(dimacsSolver);
                    dr.parseInstance(new FileReader(fmFile));
                }
                //System.out.println(p.size());
                for (Integer i : p) {
                    t.push(i);
                    //System.out.println(i);
                }



                System.out.println(dimacsSolver.isSatisfiable(t));

                System.out.println("");
                for (int i = 1; i <= 6888; i++) {
                    if (!p.contains(i) && !p.contains(-i)) {
                        System.out.println(featuresMap.get(i));
                    }


                }
                System.exit(0);
            }

            List<String> f = new ArrayList<String>();

            for (int i = 1; i <= featuresMap.size(); i++) {
                IVecInt prod = new VecInt();
                prod.push(i);
                if (!dimacsSolver.isSatisfiable(prod)) {
//                    IVecInt prod2 = new VecInt();
//                    prod2.push(i);
//                    if (reasonerSAT.getSolver().isSatisfiable(prod2)) {
//                       // f.add(featuresMap.get(i));
                    System.out.println(featuresMap.get(i));
//                    }


                }

            }

            //System.out.println(f.equals(reasonerSAT.allFreeFeatures(null)));
        } catch (Exception ex) {
            Logger.getLogger(SPL.class.getName()).log(Level.SEVERE, null, ex);
            System.out.println(
                    "error");
        }
    }

//    public boolean containPair(String file, FeaturesPair pair) {
//        try {
//            BufferedReader in = new BufferedReader(new FileReader(file));
//            String line;
//
//            while ((line = in.readLine()) != null) {
//                if (line.equals(pair.getX() + ";" + pair.getY()) || line.equals(pair.getY() + ";" + pair.getX())) {
//                    in.close();
//                    return true;
//                }
//            }
//
//            in.close();
//
//
//        } catch (Exception ex) {
//            Logger.getLogger(SPL.class.getName()).log(Level.SEVERE, null, ex);
//        }
//        return false;
//    }
//    public void computeProductPairs(String fmFile, boolean dimacs, String conf) {
//
//        try {
//            if (!new File(fmFile).exists()) {
//                throw new ParameterException("The specified FM file does not exist");
//            }
//
//            this.predictable = false;
//            this.dimacs = dimacs;
//            this.dimacsFile = fmFile;
//            if (!dimacs) {
//                fm = loadFeatureModel(fmFile);
//                fm.loadModel();
//                reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
//                reasonerSAT.init();
//            } else {
//                dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
//                dimacsSolver.setTimeout(SATtimeout);
//                DimacsReader dr = new DimacsReader(dimacsSolver);
//                dr.parseInstance(new FileReader(fmFile));
//            }
//
//
//
//
//            List<Integer> featuresList = new ArrayList<Integer>();
//            Map<Integer, String> featuresMap = new HashMap<Integer, String>();
//            Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
//
//            if (!dimacs) {
//                computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
//            } else {
//                computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
//            }
//
//
//            List<Integer> prodFeat = new ArrayList<Integer>();
//
//
//            BufferedReader in = new BufferedReader(new FileReader(conf));
//            String line;
//
//            while ((line = in.readLine()) != null) {
//                if (!line.isEmpty() && !line.contains("#")) {
//                    String feature = line.substring(0, line.indexOf("="));
//                    feature = feature.substring(feature.indexOf("_") + 1, feature.length());
//                    String val = line.substring(line.indexOf("=") + 1, line.length()).trim();
//                    int nf = featuresMapRev.get(feature);
//                    if (!val.equals("\"\"")) {
//                        prodFeat.add(nf);
//                    } else {
//                        prodFeat.add(-nf);
//                    }
//                }
//
//            }
//
//
////pairs on partial
//
//            for (int i = 0; i < prodFeat.size(); i++) {
//                for (int j = 0; j < prodFeat.size(); j++) {
//                    if (j > i) {
//                        int left = prodFeat.get(i);
//                        int right = prodFeat.get(j);
//                        if (Math.abs(left) != Math.abs(right)) {
//                            FeaturesPair pair = new FeaturesPair(left, right);
//                            //pairs.add(pair);
//                            //if (!containPair("/home/chris/2.6.28.6-icse11.dimacs.validpairs", pair)) {
//                            System.out.println(pair.getX() + ";" + pair.getY());
//                            //}
//
//                        }
//                    }
//
//                }
//
//            }
//            in.close();
//
//
//
//        } catch (Exception ex) {
//            Logger.getLogger(SPL.class.getName()).log(Level.SEVERE, null, ex);
//            System.out.println(
//                    "error");
//        }
//    }
    public void isDimacsValid(String fmFile, boolean dimacs, String dirconf) {

        try {
            if (!new File(fmFile).exists()) {
                throw new ParameterException("The specified FM file does not exist");
            }

            this.predictable = false;
            this.dimacs = dimacs;
            this.dimacsFile = fmFile;
            if (!dimacs) {
                fm = loadFeatureModel(fmFile);
                fm.loadModel();
                reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
                reasonerSAT.init();
            } else {
                dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                dimacsSolver.setTimeout(SATtimeout);
                DimacsReader dr = new DimacsReader(dimacsSolver);
                dr.parseInstance(new FileReader(fmFile));
            }




            List<Integer> featuresList = new ArrayList<Integer>();
            Map<Integer, String> featuresMap = new HashMap<Integer, String>();
            Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();

            if (!dimacs) {
                computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
            } else {
                computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
            }


            File[] confs = new File(dirconf).listFiles();

            for (File conf : confs) {

                BufferedReader in = new BufferedReader(new FileReader(conf));
                String line;
                List<Integer> integs = new ArrayList<Integer>();

                IVecInt prod = new VecInt(6888);
                while ((line = in.readLine()) != null) {
                    if (!line.isEmpty() && !line.contains("#")) {
                        String feature = line.substring(0, line.indexOf("="));
                        feature = feature.substring(feature.indexOf("_") + 1, feature.length());
                        String val = line.substring(line.indexOf("=") + 1, line.length()).trim();
                        int nf = featuresMapRev.get(feature);
                        if (!val.equals("\"\"")) {
                            prod.push(nf);
                        } else {
                            prod.push(-nf);
                        }
                        integs.add(Math.abs(nf));

                    }
//                Integer i = Integer.parseInt(line);
//
////                if (i == 1){
////                    prod.push(-i);
////                integs.add((Integer) ((int) Math.abs(-i)));
////                }
////                else{
//                prod.push(i);
//                integs.add((Integer) ((int) Math.abs(i)));
//                System.out.println(featuresMap.get((int) Math.abs(i)));
//                }

//                System.out.println(i + "->" + dimacsSolver.isSatisfiable(prod));
                }

//                for (int i = 1; i <= 6888; i++) {
//                    if (!integs.contains(i)) {
//                        prod.push(-i);
//                    }
//
//                }

                //System.out.println(prod.size());
                in.close();
                System.out.println(conf.getName() + ":" + dimacsSolver.isSatisfiable(prod));


            }
        } catch (Exception ex) {
            Logger.getLogger(SPL.class.getName()).log(Level.SEVERE, null, ex);
            System.out.println(
                    "error");
        }

    }

    public IVecInt productToInt(Product p) {
        IVecInt prod = new VecInt(p.size() - 1);
        List<Integer> productList = GA.productToList(p);
        int j = 0;
        for (Integer n : productList) {
            if (j++ <= p.size() / 2) {
                prod.push(n);
            }

        }
        return prod;
    }

    public void computePrioritizationSPLCATSolverProducts(String splcatFile, String fmFile, String outputDir, int runs, String validPairsFile, boolean predictable) throws Exception {
        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }
        File splcatCSV = new File(splcatFile);
        if (!splcatCSV.exists()) {
            throw new ParameterException("The specified SPLCAT file does not exist");
        }
        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM (xml) file does not exist");
        }

        if (validPairsFile != null && !new File(validPairsFile).exists()) {
            throw new ParameterException("The specified valid pairs file does not exist");
        }

        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        fm = loadFeatureModel(fmFile);
        fm.loadModel();
        reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
        reasonerSAT.init();

        this.predictable = predictable;


        String[] features = reasonerSAT.getVarIndex2NameMap();
        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
        computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);



        String splcatFileName = splcatCSV.getName();

        List<Product> splcatProducts = loadProductsFromCSVFile(splcatCSV, featuresMapRev);
        int splcatProductsSize = splcatProducts.size();
        int productsSize = splcatProducts.size() * 2;

        SimilarityTechnique simJaccardGreedy = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.GREEDY_SEARCH);
        SimilarityTechnique simJaccardOpti = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH);
        RandomTechnique random = new RandomTechnique();

        double[] coverageValuesSolver = new double[productsSize];
        double[] fitnessValuesSolver = new double[productsSize];
        double[] coverageValuesSimGreedy = new double[productsSize];
        double[] fitnessValuesSimGreedy = new double[productsSize];
        double[] coverageValuesSimOpti = new double[productsSize];
        double[] fitnessValuesSimOpti = new double[productsSize];

        shuffle(splcatProducts);
        reasonerSAT.init();
        if (!predictable) {
            ((Solver) reasonerSAT.getSolver()).setOrder(order);
        }
        solverIterator = new ModelIterator(reasonerSAT.getSolver());
        solverIterator.setTimeoutMs(iteratorTimeout);

        Set<TSet> validPairs;

        if (validPairsFile != null) {
            validPairs = loadPairs(validPairsFile);
        } else {
            validPairs = computePairsCoveredByProducts(splcatProducts);
        }
        for (int i = 0; i < runs; i++) {

            // SPLCAT + solver products

            List<Product> solverProducts;
            if (!predictable) {
                solverProducts = getUnpredictableProducts(splcatProductsSize);
            } else {
                solverProducts = getPredictableProducts(splcatProductsSize, features.length);
            }
            List<Product> allProducts = new ArrayList<Product>(splcatProducts);
            allProducts.addAll(solverProducts);

            shuffle(allProducts);


            // Solver
            List<Product> solverPrioritized = random.prioritize(allProducts);
            computeProductsCoverage(solverPrioritized, validPairs);
            for (int j = 0; j < solverPrioritized.size(); j++) {
                coverageValuesSolver[j] += solverPrioritized.get(j).getCoverage();
            }

            double[] fitnessSolver = computeFitnessSums(solverPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSolver.length; j++) {
                fitnessValuesSolver[j] += fitnessSolver[j];

            }

            // Similarity (Greedy)
            List<Product> simGreedyPrioritized = simJaccardGreedy.prioritize(allProducts);
            computeProductsCoverage(simGreedyPrioritized, validPairs);
            for (int j = 0; j < simGreedyPrioritized.size(); j++) {
                coverageValuesSimGreedy[j] += simGreedyPrioritized.get(j).getCoverage();
            }
            double[] fitnessSimGreedy = computeFitnessSums(simGreedyPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSimGreedy.length; j++) {
                fitnessValuesSimGreedy[j] += fitnessSimGreedy[j];

            }

            // Similarity (Near optimal)
            List<Product> simOptiPrioritized = simJaccardOpti.prioritize(allProducts);
            computeProductsCoverage(simOptiPrioritized, validPairs);
            for (int j = 0; j < simOptiPrioritized.size(); j++) {
                coverageValuesSimOpti[j] += simOptiPrioritized.get(j).getCoverage();
            }
            double[] fitnessSimOpti = computeFitnessSums(simOptiPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSimOpti.length; j++) {
                fitnessValuesSimOpti[j] += fitnessSimOpti[j];

            }

        }
        for (int i = 0; i < coverageValuesSolver.length; i++) {
            coverageValuesSolver[i] /= runs;
            coverageValuesSimGreedy[i] /= runs;
            coverageValuesSimOpti[i] /= runs;
            fitnessValuesSolver[i] /= runs;
            fitnessValuesSimGreedy[i] /= runs;
            fitnessValuesSimOpti[i] /= runs;
        }

        String sSolver = predictable ? "Predictable-" : "Unpredictable-";
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProductsAndSolver-" + sSolver + runs + "runs.dat", coverageValuesSolver, fitnessValuesSolver);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProductsAndSolver-SimJaccardGreedy-" + runs + "runs.dat", coverageValuesSimGreedy, fitnessValuesSimGreedy);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProductsAndSolver-SimJaccardOpti-" + runs + "runs.dat", coverageValuesSimOpti, fitnessValuesSimOpti);

    }

    public void computePrioritizationSolverProducts(String fmFile, String outputDir, int runs, String validPairsFile, boolean predictable) throws Exception {

        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }
        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (validPairsFile != null && !new File(validPairsFile).exists()) {
            throw new ParameterException("The specified valid pairs file does not exist");
        }

        String fmFileName = new File(fmFile).getName();
        fm = loadFeatureModel(fmFile);
        fm.loadModel();
        reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
        reasonerSAT.init();

        String[] features = reasonerSAT.getVarIndex2NameMap();
        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
        computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);

        reasonerSAT.init();


        this.predictable = predictable;

        Set<TSet> validPairs;
        if (validPairsFile != null) {
            validPairs = loadPairs(validPairsFile);
        } else {
            validPairs = computeValidPairs(featuresMap, featuresList, null, false, null, 1, 1);
        }
        int featuresCount = features.length;
        int productsSize = featuresCount / 2;

        double[] coverageValuesSolver = new double[productsSize];
        double[] fitnessValuesSolver = new double[productsSize];
        double[] coverageValuesSimGreedy = new double[productsSize];
        double[] fitnessValuesSimGreedy = new double[productsSize];
        double[] coverageValuesSimOpti = new double[productsSize];
        double[] fitnessValuesSimOpti = new double[productsSize];

        SimilarityTechnique simJaccardGreedy = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.GREEDY_SEARCH);
        SimilarityTechnique simJaccardOpti = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH);

        reasonerSAT.init();
        if (!predictable) {
            ((Solver) reasonerSAT.getSolver()).setOrder(order);
        }
        solverIterator = new ModelIterator(reasonerSAT.getSolver());
        solverIterator.setTimeoutMs(iteratorTimeout);

        for (int i = 0; i < runs; i++) {

            // Solver
            List<Product> solverProducts;
            if (!predictable) {
                solverProducts = getUnpredictableProducts(productsSize);
            } else {
                solverProducts = getPredictableProducts(productsSize, features.length);
            }


            shuffle(solverProducts);
            computeProductsCoverage(solverProducts, validPairs);
            for (int j = 0; j < solverProducts.size(); j++) {
                coverageValuesSolver[j] += solverProducts.get(j).getCoverage();
            }
            double[] fitnessSolver = computeFitnessSums(solverProducts, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSolver.length; j++) {
                fitnessValuesSolver[j] += fitnessSolver[j];

            }

            // Similarity (Greedy)
            List<Product> simGreedyPrioritized = simJaccardGreedy.prioritize(solverProducts);
            // The pairs covered are the same
            computeProductsCoverage(simGreedyPrioritized, validPairs);

            for (int j = 0; j < simGreedyPrioritized.size(); j++) {
                coverageValuesSimGreedy[j] += simGreedyPrioritized.get(j).getCoverage();
            }
            double[] fitnessSimGreedy = computeFitnessSums(simGreedyPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSimGreedy.length; j++) {
                fitnessValuesSimGreedy[j] += fitnessSimGreedy[j];

            }

            // Similarity (Near optimal)
            List<Product> simOptiPrioritized = simJaccardOpti.prioritize(solverProducts);
            // The pairs covered are the same
            computeProductsCoverage(simOptiPrioritized, validPairs);
            for (int j = 0; j < simOptiPrioritized.size(); j++) {
                coverageValuesSimOpti[j] += simOptiPrioritized.get(j).getCoverage();
            }
            double[] fitnessSimOpti = computeFitnessSums(simOptiPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSimOpti.length; j++) {
                fitnessValuesSimOpti[j] += fitnessSimOpti[j];

            }
        }

        for (int i = 0; i < coverageValuesSolver.length; i++) {
            coverageValuesSolver[i] /= runs;
            coverageValuesSimGreedy[i] /= runs;
            coverageValuesSimOpti[i] /= runs;
            fitnessValuesSolver[i] /= runs;
            fitnessValuesSimGreedy[i] /= runs;
            fitnessValuesSimOpti[i] /= runs;
        }
        String sSolver = predictable ? "Predictable-" : "Unpredictable-";
        writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_ProductsSolver-" + sSolver + runs + "runs.dat", coverageValuesSolver, fitnessValuesSolver);
        writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_ProductsSolver-SimJaccardGreedy-" + runs + "runs.dat", coverageValuesSimGreedy, fitnessValuesSimGreedy);
        writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_ProductsSolver-SimJaccardOpti-" + runs + "runs.dat", coverageValuesSimOpti, fitnessValuesSimOpti);
    }

    public FeatureModel loadFeatureModel(String fmFile) {
        return new XMLFeatureModel(fmFile, XMLFeatureModel.USE_VARIABLE_NAME_AS_ID);
    }

    public List<Product> getUnpredictableProducts(int count) throws Exception {
        List<Product> products = new ArrayList<Product>(count);

        while (products.size() < count) {

            try {
                if (solverIterator.isSatisfiable()) {
                    Product product = toProduct(solverIterator.model());

                    if (!products.contains(product)) {
                        products.add(product);
                    }

                } else {
                    if (!dimacs) {
                        reasonerSAT.init();
                        if (!predictable) {
                            ((Solver) reasonerSAT.getSolver()).setOrder(order);
                        }
                        solverIterator = new ModelIterator(reasonerSAT.getSolver());
                        solverIterator.setTimeoutMs(iteratorTimeout);

                    } else {
                        dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                        dimacsSolver.setTimeout(SATtimeout);
                        DimacsReader dr = new DimacsReader(dimacsSolver);
                        dr.parseInstance(new FileReader(dimacsFile));
                        if (!predictable) {
                            ((Solver) dimacsSolver).setOrder(order);
                        }
                        solverIterator = new ModelIterator(dimacsSolver);
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    }
                }
            } catch (TimeoutException e) {
            }
        }
        return products;
    }

    public Product getUnpredictableProduct() throws Exception {
        Product product = null;
        while (product == null) {
            try {
                if (solverIterator.isSatisfiable()) {
                    product = toProduct(solverIterator.model());
                } else {
                    if (!dimacs) {
                        reasonerSAT.init();
                        if (!predictable) {
                            ((Solver) reasonerSAT.getSolver()).setOrder(order);
                        }
                        solverIterator = new ModelIterator(reasonerSAT.getSolver());
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    } else {
                        dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                        dimacsSolver.setTimeout(SATtimeout);
                        DimacsReader dr = new DimacsReader(dimacsSolver);
                        dr.parseInstance(new FileReader(dimacsFile));
                        if (!predictable) {
                            ((Solver) dimacsSolver).setOrder(order);
                        }
                        solverIterator = new ModelIterator(dimacsSolver);
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    }
                }
            } catch (TimeoutException e) {
            }
        }
        return product;
    }

    public List<Product> getPredictableProducts(int count, int numberOfFeatures) throws Exception {
        List<Product> products = new ArrayList<Product>(count);
        while (products.size() < count) {
            try {
                if (solverIterator.isSatisfiable()) {
                    Product product = toProduct(solverIterator.model());
                    if (randomGenerator.nextInt(numberOfFeatures) == numberOfFeatures - 1) {

                        if (!products.contains(product)) {
                            products.add(product);
                        }
                    }
                } else {
                    if (!dimacs) {
                        reasonerSAT.init();
                        if (!predictable) {
                            ((Solver) reasonerSAT.getSolver()).setOrder(order);
                        }
                        solverIterator = new ModelIterator(reasonerSAT.getSolver());
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    } else {
                        dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                        dimacsSolver.setTimeout(SATtimeout);
                        DimacsReader dr = new DimacsReader(dimacsSolver);
                        dr.parseInstance(new FileReader(dimacsFile));
                        if (!predictable) {
                            ((Solver) dimacsSolver).setOrder(order);
                        }
                        solverIterator = new ModelIterator(dimacsSolver);
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    }
                }

            } catch (TimeoutException e) {
            }
        }
        return products;
    }

    public Set<TSet> computePairsCoveredByProducts(List<Product> products) {
        Set<TSet> coveredPairs = new HashSet<TSet>();
        for (Product product : products) {
            Set<TSet> productPairs = product.computeCoveredPairs();
            coveredPairs.addAll(productPairs);
        }
        return coveredPairs;
    }

    public void computeProductsPartialCoverage(List<Product> products, Set<TSet> pairs) {
        double pairsSize = pairs.size();
        Set<TSet> pairsCopy = new HashSet<TSet>(pairs);
        for (Product product : products) {
            int initialSize = pairsCopy.size();
            Set<TSet> productPairs = product.computePartialCoveredPairs(pairsCopy);
            pairsCopy.removeAll(productPairs);
            double removed = initialSize - pairsCopy.size();
            double coverage = removed / pairsSize * 100.0;
            product.setCoverage(coverage);
        }
        pairsCopy = null;
    }

    public void computeProductsCoverage(List<Product> products, Set<TSet> pairs) {
        double pairsSize = pairs.size();
        Set<TSet> pairsCopy = new HashSet<TSet>(pairs);
        for (Product product : products) {
            int initialSize = pairsCopy.size();
            Set<TSet> productPairs = product.computeCoveredPairs();
            pairsCopy.removeAll(productPairs);
            double removed = initialSize - pairsCopy.size();
            double coverage = removed / pairsSize * 100.0;
            product.setCoverage(coverage);
        }
        pairsCopy = null;
    }

    public void shuffle(List<Product> products) {
        List<Product> productsCopy = new ArrayList<Product>(products);
        int done = 0;
        while (done < products.size()) {
            int index = randomGenerator.nextInt(productsCopy.size());
            products.set(done++, productsCopy.get(index));
            productsCopy.remove(index);
        }
    }

    public void writeDimacsProductToFile(String outFile, Product product) throws Exception {
        BufferedWriter out = new BufferedWriter(new FileWriter(outFile));

        for (Integer i : product) {
            out.write(Integer.toString(i));
            //if (n < product.size()) {
            out.newLine();
            //}
        }
        out.close();
    }

    public Set<TSet> loadSerializedTSets(String inFile) {


        Set<TSet> tsets = null;
        try {



            FileInputStream fileIn = new FileInputStream(inFile);
            ObjectInputStream in = new ObjectInputStream(fileIn);

            tsets = (HashSet<TSet>) in.readObject();

            in.close();
            fileIn.close();

        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return tsets;

    }

    public List<Product> loadSerializedProducts(String inFile) {


        List<Product> prods = null;
        try {



            FileInputStream fileIn = new FileInputStream(inFile);
            ObjectInputStream in = new ObjectInputStream(fileIn);

            prods = (ArrayList<Product>) in.readObject();

            in.close();
            fileIn.close();

        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return prods;

    }

    public void serializeTSets(String outFile, Set<TSet> tsets) {
        try {


            FileOutputStream fileOut = new FileOutputStream(outFile);
            ObjectOutputStream out = new ObjectOutputStream(fileOut);

            out.writeObject(tsets);
            out.close();
            fileOut.close();

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void serializeProducts(String outFile, List<Product> products) {
        try {


            FileOutputStream fileOut = new FileOutputStream(outFile);
            ObjectOutputStream out = new ObjectOutputStream(fileOut);

            out.writeObject(products);
            out.close();
            fileOut.close();

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void writeProductsToFile(String outFile, List<Product> products, Map<Integer, String> featuresMap, List<Integer> featuresList) throws Exception {
//        BufferedWriter out = new BufferedWriter(new FileWriter(outFile));

//        out.write("Feature\\Product;");
//
//        for (int i = 0; i < products.size(); i++) {
//            out.write("" + i + ";");
//        }
//
//        out.newLine();
//
//        int featuresCount = featuresList.size() / 2;
//        for (int i = 1; i <= featuresCount; i++) {
//            out.write(featuresMap.get(i) + ";");
//
//            for (Product p : products) {
//                for (Integer n : p) {
//                    if (Math.abs(n) == i) {
//                        if (n > 0) {
//                            out.write("X;");
//                        } else {
//                            out.write("-;");
//                        }
//                    }
//                }
//            }
//            out.newLine();
//        }
//        out.close();


        BufferedWriter out = new BufferedWriter(new FileWriter(outFile));

        int featuresCount = featuresList.size() / 2;
        for (int i = 1; i <= featuresCount; i++) {
            out.write(i + ":" + featuresMap.get(i));
            if (i < featuresCount) {
                out.write(";");
            }
        }
        out.newLine();
        for (Product product : products) {
            List<Integer> prodFeaturesList = new ArrayList<Integer>(product);
            Collections.sort(prodFeaturesList, new Comparator<Integer>() {

                @Override
                public int compare(Integer o1, Integer o2) {
                    return ((Integer) Math.abs(o1)).compareTo(((Integer) Math.abs(o2)));
                }
            });

            int done = 1;
            for (Integer feature : prodFeaturesList) {
                out.write((feature > 0 ? "X" : "-"));
                if (done < featuresCount) {
                    out.write(";");
                }
                done++;
            }

            out.newLine();
        }
        out.close();
    }

    public boolean isValidProduct(Product product, Map<Integer, String> featuresMap, List<Integer> featuresList) throws Exception {
        IVecInt prod = new VecInt(product.size());

        for (Integer s : product) {

            if (s < 0) {
                prod.push(-reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get((-s) - 1))));
            } else {
                prod.push(reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get(s - 1))));
            }
        }
        return reasonerSAT.getSolver().isSatisfiable(prod);
    }

    public boolean isValidPair(TSet pair, Map<Integer, String> featuresMap, List<Integer> featuresList) throws Exception {

        IVecInt prod = new VecInt(2);


        for (Integer fi : pair.getVals()) {
            if (!dimacs) {
                if (fi < 0) {
                    prod.push(-reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get((-fi) - 1))));
                } else {
                    prod.push(reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get(fi - 1))));
                }
            } else {
                prod.push(fi);


            }
        }
        if (!dimacs) {
            return reasonerSAT.getSolver().isSatisfiable(prod);
        } else {
            return dimacsSolver.isSatisfiable(prod);
        }

    }

    private Set<TSet> computeNRandValidTSets(List<Product> products, int n, int t) throws Exception {
        Set<TSet> pairs = new HashSet<TSet>(n);
        Random r = new Random();
        while (pairs.size() < n) {
            TSet set = new TSet();
            Product p = products.get(r.nextInt(products.size()));
            List<Integer> prod = new ArrayList<Integer>(p);
            while (set.getSize() < t) {
                set.add(prod.get(r.nextInt(prod.size())));
            }
            pairs.add(set);
        }
        return pairs;
    }

    private Set<TSet> computeNRandValidTSets(Map<Integer, String> featuresMap, List<Integer> featuresList, int n, int t) throws Exception {



        Set<TSet> pairs = new HashSet<TSet>(n);

        int size = featuresList.size();
        double total = getBinomCoeff(size, t);
        while (pairs.size() < n) {
            TSet set = getITSet(size, t, Math.floor(Math.random() * total), featuresList);
            if (isValidPair(set, featuresMap, featuresList)) {
                pairs.add(set);
            }
        }

        return pairs;
    }

    private Set<TSet> computeValidPairs(Map<Integer, String> featuresMap, List<Integer> featuresList,
            String outFile, boolean dimacs, ISolver dimacsSolver, int nbParts, int part) throws Exception {


        if (part > nbParts || nbParts < 1) {
            throw new ParameterException("Invalid parts parameters");
        }


        Set<TSet> pairs = new HashSet<TSet>();





        int size = featuresList.size();

        nCk(size, 2, pairs, featuresMap, featuresList);
        //System.out.println(pairs);
        //System.out.println(pairs.size());
        return pairs;
    }

    public void computeFeatures(ReasoningWithSAT reasonerSAT, Map<Integer, String> featuresMap, Map<String, Integer> featuresMapRev, List<Integer> featuresList, boolean dimacs, String dimacsFile) throws Exception {

        if (!dimacs) {
            String[] features = reasonerSAT.getVarIndex2NameMap();

            for (int i = 0; i < features.length; i++) {
                String feature = features[i];
                int n = i + 1;
                featuresList.add(n);
                featuresMap.put(n, feature);
                featuresMapRev.put(feature, n);
            }


        } else {
            BufferedReader in = new BufferedReader(new FileReader(dimacsFile));
            String line;
            int n = 0;
            while ((line = in.readLine()) != null && line.startsWith("c")) {
                StringTokenizer st = new StringTokenizer(line.trim(), " ");
                st.nextToken();
                n++;
                String sFeature = st.nextToken().replace('$', ' ').trim();
                int feature = Integer.parseInt(sFeature);
                if (n != feature) {
                    throw new Exception("Incorrect dimacs file, missing feature number " + n + " ?");
                }
                String featureName = st.nextToken();
                featuresList.add(feature);
                featuresMap.put(feature, featureName);
                featuresMapRev.put(featureName, feature);
            }
            in.close();
        }

        int n = 1;
        int featuresCount = featuresList.size();
        while (n <= featuresCount) {
            featuresList.add(-n);
            n++;
        }


    }

    public void writeCoverageAndFitnessValuesToFile(String outFile, double[] coverageValues, double[] fitnessSums) throws IOException {
        BufferedWriter out = new BufferedWriter(new FileWriter(outFile));
        out.write("#coverage of pairs (in %, starting from 0 products selected);Fitness func");
        out.newLine();
        out.write("0;0");
        out.newLine();
        double s = 0;
        for (int i = 0; i < coverageValues.length; i++) {
            s += coverageValues[i];
            out.write(Double.toString(s) + ";" + Double.toString(fitnessSums[i]));
            out.newLine();
        }
        out.close();
    }

    public void normalizeDataFile(String inputDir) throws Exception {

        File inDir = new File(inputDir);
        if (!inDir.exists()) {
            throw new ParameterException("Input directory does not exist");
        }

        File[] datFiles = inDir.listFiles(new FilenameFilter() {

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".dat") && !name.toLowerCase().contains("norm");
            }
        });

        for (File file : datFiles) {

            int count = countUncommentedLines(file);

            double[] coverageValues = new double[count];
            double[] fitnessValues = new double[count];

            BufferedReader in = new BufferedReader(new FileReader(file));

            int i = 0;
            String line;

            while ((line = in.readLine()) != null) {
                line = line.trim();
                if (!line.startsWith("#")) {
                    StringTokenizer st = new StringTokenizer(line, ";");

                    coverageValues[i] = Double.parseDouble(st.nextToken().trim());
                    fitnessValues[i] = Double.parseDouble(st.nextToken().trim());
                    i++;
                }
            }
            in.close();

            double[] normalizedCoverageValues = new double[101];
            double[] normalizedFitnessValues = new double[101];

            for (int j = 0; j < normalizedCoverageValues.length; j++) {
                int prodIndex = (int) ((double) j / 100.0 * (coverageValues.length - 1));
                normalizedCoverageValues[j] = coverageValues[prodIndex];
                normalizedFitnessValues[j] = fitnessValues[prodIndex] / fitnessValues[fitnessValues.length - 1] * 100;
            }


            String outFile = file.toString().replace(".dat", "-Norm.dat");
            BufferedWriter out = new BufferedWriter(new FileWriter(outFile));
            out.write("#coverage of pairs (in %, starting from 0% of products selected (normalized));Fitness func (normalized)");
            out.newLine();
            for (int k = 0; k < normalizedCoverageValues.length; k++) {
                out.write(Double.toString(normalizedCoverageValues[k]) + ";" + Double.toString(normalizedFitnessValues[k]));
                out.newLine();
            }
            out.close();
        }

    }

    public Product toProduct(int[] vector) {

        Product product = new Product();
        for (int i : vector) {
            product.add(i);
        }
        return product;
    }

    public void computeAverageDataFiles(String inputDir, String outputDir, final boolean noNorm) throws Exception {
        File inDir = new File(inputDir);
        if (!inDir.exists()) {
            throw new ParameterException("Input directory does not exist");
        }

        if (outputDir.equals("Same as input")) {
            outputDir = inputDir;
        }

        if (!new File(outputDir).exists()) {
            throw new ParameterException("Output directory does not exist");
        }
        File[] datFiles = inDir.listFiles(new FilenameFilter() {

            @Override
            public boolean accept(File dir, String name) {
                if (!noNorm) {
                    return name.endsWith("-Norm.dat");
                } else {
                    return name.endsWith(".dat") && !name.toLowerCase().contains("norm");
                }
            }
        });

        Set<String> types = new HashSet<String>();
        for (File file : datFiles) {
            String sFile = file.toString();
            String type = sFile.substring(sFile.lastIndexOf("_") + 1, sFile.length());
            types.add(type);
        }
        for (final String type : types) {
            datFiles = inDir.listFiles(new FilenameFilter() {

                @Override
                public boolean accept(File dir, String name) {
                    return name.endsWith(type);
                }
            });
            int n = 0;
            double[] coverageValues, fitnessValues;
            if (!noNorm) {
                coverageValues = new double[101];
                fitnessValues = new double[101];
            } else {
                int count = minUncommentedLinesCount(datFiles);
                coverageValues = new double[count];
                fitnessValues = new double[count];
            }

            String firstLine = "";
            for (File dat : datFiles) {
                int i = 0;
                BufferedReader in = new BufferedReader(new FileReader(dat));
                String line;
                while ((line = in.readLine()) != null && i < coverageValues.length) {
                    line = line.trim();
                    if (!line.isEmpty()) {
                        if (line.startsWith("#")) {
                            firstLine = line;
                        } else {
                            StringTokenizer tokenizer = new StringTokenizer(line, ";");
                            double cov = Double.parseDouble(tokenizer.nextToken());
                            double fit = Double.parseDouble(tokenizer.nextToken());
                            coverageValues[i] += cov;
                            fitnessValues[i] += fit;
                            i++;
                        }
                    }
                }
                in.close();
                n++;

            }

            for (int i = 0; i < coverageValues.length; i++) {
                coverageValues[i] /= (double) n;
                fitnessValues[i] /= (double) n;
            }

            String outFile = outputDir;
            if (!outFile.endsWith("/")) {
                outFile += "/";
            }
            outFile = outFile + "AVG_ON_ALL_" + type;
            BufferedWriter out = new BufferedWriter(new FileWriter(outFile));

            out.write(firstLine);
            out.newLine();
            for (int i = 0; i < coverageValues.length; i++) {
                out.write(Double.toString(coverageValues[i]) + ";" + Double.toString(fitnessValues[i]));
                out.newLine();
            }
            out.close();
        }
    }

    public int countUncommentedLines(File file) throws Exception {
        BufferedReader in = new BufferedReader(new FileReader(file));
        String line;
        int n = 0;
        while ((line = in.readLine()) != null) {
            line = line.trim();
            if (!line.isEmpty() && !line.startsWith("#")) {
                n++;
            }
        }
        in.close();
        return n;
    }

    public int minUncommentedLinesCount(File[] files) throws Exception {
        int min = countUncommentedLines(files[0]);

        for (int i = 1; i < files.length; i++) {
            int count = countUncommentedLines(files[i]);
            if (count < min) {
                min = count;
            }
        }

        return min;
    }

    public List<Product> loadProductsFromCSVFile(File csvFile, Map<String, Integer> featuresMapRev) throws Exception {
        List<Product> products = new ArrayList<Product>();
        BufferedReader in = new BufferedReader(new FileReader(csvFile));
        String line;
        boolean firstLine = true;
        List<String> features = null;

        if (featuresMapRev != null) {
            features = new ArrayList<String>();
        }
        while ((line = in.readLine()) != null) {
            StringTokenizer tokenizer = new StringTokenizer(line, ";");
            if (firstLine) {
                if (featuresMapRev != null) {
                    while (tokenizer.hasMoreTokens()) {
                        features.add(tokenizer.nextToken().trim());
                    }
                }
                firstLine = false;
            } else {
                Product product = new Product();
                int count;
                if (featuresMapRev != null) {
                    count = 0;
                } else {
                    count = 1;
                }
                while (tokenizer.hasMoreTokens()) {
                    String tok = tokenizer.nextToken().trim();
                    if (tok.equals("X")) {
                        if (featuresMapRev != null) {
                            product.add(featuresMapRev.get(features.get(count)));
                        } else {
                            product.add(count);
                        }
                    } else if (tokenizer.equals("-")) {
                        if (featuresMapRev != null) {
                            product.add(-featuresMapRev.get(features.get(count)));
                        } else {
                            product.add(-count);
                        }
                    }
                    count++;

                }
                products.add(product);
            }
        }
        return products;
    }

    public double[] computeFitnessSums(List<Product> products, int distanceMethod) {
        int size = products.size();

        double[][] distancesMatrix = new double[size][size];

        for (int i = 0; i < size; i++) {
            for (int j = 0; j < size; j++) {
                if (j > i) {
                    distancesMatrix[i][j] = DistancesUtil.getJaccardDistance(products.get(i), products.get(j));
                }
            }
        }
        double[] fitnessSums = new double[size];
        int n = size - 1;

        while (n >= 0) {
            fitnessSums[n] = SimilarityTechnique.getJaccardFitnessSum(distancesMatrix, n + 1);
            n--;
        }
        return fitnessSums;


    }

    public void computeValidPairsToFile(String fmFile, boolean dimacs, int nbParts, int part) throws Exception {

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }


        int timeoutS = 300;
        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
        if (!dimacs) {

            fm = loadFeatureModel(fmFile);
            fm.loadModel();
            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, timeoutS);
            reasonerSAT.init();
            reasonerSAT.getSolver().setTimeout(timeoutS);


            computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);

            computeValidPairs(featuresMap, featuresList, (fmFile + ".validpairs"), false, null, nbParts, part);
        } else {
            computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
            dimacsSolver.setTimeout(timeoutS);
            DimacsReader dr = new DimacsReader(dimacsSolver);
            dr.parseInstance(new FileReader(fmFile));
            computeValidPairs(featuresMap, featuresList, (fmFile + ".validpairs"), true, dimacsSolver, nbParts, part);
        }
    }

    public Set<TSet> loadPairs(String pairsFile) throws Exception {
        if (!new File(pairsFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }

        LineNumberReader lnr = new LineNumberReader(new FileReader(pairsFile));
        lnr.skip(Long.MAX_VALUE);


        List<TSet> pairs = new ArrayList<TSet>(lnr.getLineNumber());

//        BufferedReader in = new BufferedReader(new FileReader(pairsFile));
//        String line;
//
//        while ((line = in.readLine()) != null) {
//            if (!line.isEmpty()) {
//                StringTokenizer st = new StringTokenizer(line, ";");
//
//                FeaturesPair pair = new FeaturesPair(Integer.parseInt(st.nextToken()), Integer.parseInt(st.nextToken()));
//                pairs.add(pair);
//            }
//        }
//
//        in.close();
//
        Set<TSet> pairsSet = new HashSet<TSet>(pairs);
        return pairsSet;





    }

    private class CommandLineParser {

        private JCommander jCommander;
        private PrioritizationSolverProducts commandPrioritizationSolverProducts;
        private AverageDataFiles commandAverageDataFiles;
        private PrioritizationSPLCATProducts commandPrioritizationSPLCATProducts;
        private PrioritizationSPLCATSolverProducts commandPrioritizationSPLCATSolverProducts;
        private NormalizeDataFiles commandNormalizeDataFiles;
        private GenerateGA commandGenerateGA;
        private ComputePairs commandComputePairs;
        private ComputeStats commandComputeStats;
        private PrioritizationGenTSets commandPrioritizationProducts;
        private String[] args;
        public static final String PRIORITIZATION_SOLVER_PRODUCTS = "prio_solver_products";
        public static final String AVERAGE_NORMALIZED_DATA_FILES = "average_data_files";
        public static final String PRIORITIZATION_SPLCAT_PRODUCTS = "prio_splcat_products";
        public static final String PRIORITIZATION_SPLCAT_SOLVER_PRODUCTS = "prio_splcat_solver_products";
        public static final String NORMALIZE_DATA_FILES = "normalize_data_files";
        public static final String GENERATE_GA = "generate_prods_ga";
        public static final String COMPUTE_PAIRS = "compute_valid_pairs";
        public static final String COMPUTE_STATS = "compute_stats";
        public static final String PRIORITIZATION_PRODUCTS = "tsets";

        public CommandLineParser(String[] args, String programName) {
            this.args = args;
            commandPrioritizationSolverProducts = new PrioritizationSolverProducts();
            commandAverageDataFiles = new AverageDataFiles();
            commandPrioritizationSPLCATProducts = new PrioritizationSPLCATProducts();
            commandPrioritizationSPLCATSolverProducts = new PrioritizationSPLCATSolverProducts();
            commandNormalizeDataFiles = new NormalizeDataFiles();
            commandGenerateGA = new GenerateGA();
            commandComputePairs = new ComputePairs();
            commandComputeStats = new ComputeStats();
            commandPrioritizationProducts = new PrioritizationGenTSets();
            jCommander = new JCommander();
            jCommander.addCommand(PRIORITIZATION_SPLCAT_PRODUCTS, commandPrioritizationSPLCATProducts);
            jCommander.addCommand(PRIORITIZATION_SOLVER_PRODUCTS, commandPrioritizationSolverProducts);
            jCommander.addCommand(PRIORITIZATION_SPLCAT_SOLVER_PRODUCTS, commandPrioritizationSPLCATSolverProducts);
            jCommander.addCommand(AVERAGE_NORMALIZED_DATA_FILES, commandAverageDataFiles);
            jCommander.addCommand(NORMALIZE_DATA_FILES, commandNormalizeDataFiles);
            jCommander.addCommand(GENERATE_GA, commandGenerateGA);
            jCommander.addCommand(COMPUTE_PAIRS, commandComputePairs);
            jCommander.addCommand(COMPUTE_STATS, commandComputeStats);
            jCommander.addCommand(PRIORITIZATION_PRODUCTS, commandPrioritizationProducts);
            jCommander.setProgramName("java -jar " + programName + ".jar");
        }

        public void parseArgs() {
            jCommander.parse(args);
        }

        public void printUsage() {
            jCommander.usage();
        }

        public String getCommandName() {
            return jCommander.getParsedCommand();
        }

        public PrioritizationSolverProducts getCommandPrioritizationSolverProducts() {
            return commandPrioritizationSolverProducts;
        }

        public AverageDataFiles getCommandAverageDataFiles() {
            return commandAverageDataFiles;
        }

        public PrioritizationSPLCATProducts getCommandPrioritizationSPLCATProducts() {
            return commandPrioritizationSPLCATProducts;
        }

        public PrioritizationSPLCATSolverProducts getCommandPrioritizationSPLCATAndSolverProducts() {
            return commandPrioritizationSPLCATSolverProducts;
        }

        public PrioritizationGenTSets getCommandPrioritizationProducts() {
            return commandPrioritizationProducts;
        }

        public NormalizeDataFiles getCommandNormalizeDataFiles() {
            return commandNormalizeDataFiles;
        }

        public GenerateGA getCommandGenerateGA() {
            return commandGenerateGA;
        }

        public ComputePairs getCommandComputePairs() {
            return commandComputePairs;
        }

        public ComputeStats getCommandComputeStats() {
            return commandComputeStats;
        }

        @Parameters(commandDescription = "Computes the values of the prioritization techniques and fitness function using the "
        + "set of valid products from SPLCAT. ")
        private class PrioritizationSPLCATProducts {

            @Parameter(names = "-csv", description = "CSV file output from SPLCAT tool", required = true)
            private String csvFile;
            @Parameter(names = "-fm", description = "Feature model (SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-o", description = "Output directory", required = true)
            private String outputFile;
            @Parameter(names = "-runs", description = "Number of runs to average the random technique")
            private int runs = 10;
            @Parameter(names = "-p", description = "Valid pairs file")
            private String validPairsFile;
        }

        @Parameters(commandDescription = "Computes the values of the prioritization techniques and fitness function using "
        + "products generated with the GA and random products. ")
        private class PrioritizationGenTSets {

            @Parameter(names = "-fm", description = "Feature model", required = true)
            private String fmFile;
            @Parameter(names = "-o", description = "Output directory", required = true)
            private String outputFile;
            @Parameter(names = "-runs", description = "Number of runs to execute")
            private int runs = 10;
            @Parameter(names = "-dimacs", description = "Specify if the FM is a dimacs one")
            private boolean dimacs = false;
            @Parameter(names = "-nbPairs", description = "Number of valid pairs to sample")
            private int nbPairs = 100000;
            @Parameter(names = "-t", description = "T-Wise")
            private int t = 3;
            @Parameter(names = "-prods", description = "Serialized products directory", required = true)
            private String prods;
            @Parameter(names = "-tsets", description = "Serialized tsets directory", required = true)
            private String pairs;
        }

        @Parameters(commandDescription = "Computes the values of the prioritization techniques and fitness function using the "
        + "set of valid products from SPLCAT + additional products from the solver. ")
        private class PrioritizationSPLCATSolverProducts {

            @Parameter(names = "-csv", description = "CSV file output from SPLCAT tool", required = true)
            private String csvFile;
            @Parameter(names = "-fm", description = "Feature model (SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-o", description = "Output directory", required = true)
            private String outputFile;
            @Parameter(names = "-runs", description = "Number of runs to average")
            private int runs = 10;
            @Parameter(names = "-p", description = "Valid pairs file")
            private String validPairsFile;
            @Parameter(names = "-predictable", description = "Use the predictable products for the solver")
            private boolean predictable = false;
        }

        @Parameters(commandDescription = "Computes the values of the prioritization techniques and fitness function using a "
        + "set of products from the solver. The number of products generated will be equal to #features/2")
        private class PrioritizationSolverProducts {

            @Parameter(names = "-fm", description = "Feature model (SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-o", description = "Output directory", required = true)
            private String outputFile;
            @Parameter(names = "-runs", description = "Number of runs to average")
            private int runs = 10;
            @Parameter(names = "-p", description = "Valid pairs file")
            private String validPairsFile;
            @Parameter(names = "-predictable", description = "Use the predictable products for the solver")
            private boolean predictable = false;
        }

        @Parameters(commandDescription = "Compute the average between the data files "
        + "specified in the directory")
        private class AverageDataFiles {

            @Parameter(names = "-i", description = "Directory containing .dat files", required = true)
            private String inputDirectory;
            @Parameter(names = "-o", description = "Output directory")
            private String outputDirectory = "Same as input";
            @Parameter(names = "-noNorm", description = "Don't work on normalized data files. The "
            + "minimum length data file will be used")
            private boolean noNorm = false;
        }

        @Parameters(commandDescription = "Compute the normalized version of the data files "
        + "present specified in the directory")
        private class NormalizeDataFiles {

            @Parameter(names = "-i", description = "Directory containing data file to normalize", required = true)
            private String input;
        }

        @Parameters(commandDescription = "Compute the valid pairs to a file for a XML FM ")
        private class ComputePairs {

            @Parameter(names = "-fm", description = "Feature model (Dimacs or SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-dimacs", description = "Specify if the FM is a dimacs one")
            private boolean dimacs = false;
            @Parameter(names = "-parts", description = "Specify the number of parts")
            private int nbParts = 1;
            @Parameter(names = "-part", description = "Specify which part should be considered. Only if nbParts > 1")
            private int part = 1;
        }

        @Parameters(commandDescription = "Compute some statistics for a FM ")
        private class ComputeStats {

            @Parameter(names = "-fm", description = "Feature model (Dimacs or SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-dimacs", description = "Specify if the FM is a dimacs one")
            private boolean dimacs = false;
        }

        @Parameters(commandDescription = "Generate products using the GA")
        private class GenerateGA {

            @Parameter(names = "-csv", description = "CSV file output from SPLCAT tool. This argument is optional, see -nbProds option")
            private String csvFile;
            @Parameter(names = "-fm", description = "Feature model (SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-o", description = "Output directory", required = true)
            private String outputFile;
            @Parameter(names = "-runs", description = "Number of runs to average")
            private int runs = 10;
            @Parameter(names = "-nbProds", description = "Number of products to generate. If negative, then "
            + "the same number as the SPLCAT output will be used. In this case, the -csv option is required")
            private int nbProds = 10;
            @Parameter(names = "-timeAllowedMS", description = "Time allowed for the GA in ms")
            private long timeAllowed = 60000;
            @Parameter(names = "-p", description = "Valid pairs file")
            private String validPairsFile;
            @Parameter(names = "-dimacs", description = "Specify if the FM is a dimacs one")
            private boolean dimacs = false;
            @Parameter(names = "-noCoverage", description = "Do not compute the coverage (only generate the products)")
            private boolean noCoverage = false;
            @Parameter(names = "-onlyGA", description = "Run only the GA approach")
            private boolean onlyGA = false;
        }
    }

    public static void main(String[] args) {
        SPL.getInstance().parseArgs(args);
    }
}
