/*
 * Author : Christopher Henard (christopher.henard@uni.lu)
 * Date : 21/05/2012
 * Copyright 2012 University of Luxembourg – Interdisciplinary Centre for Security Reliability and Trust (SnT)
 * All rights reserved
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package spl;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;
import com.beust.jcommander.Parameters;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.LineNumberReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.StringTokenizer;
import org.sat4j.core.VecInt;
import org.sat4j.minisat.SolverFactory;
import org.sat4j.minisat.core.IOrder;
import org.sat4j.minisat.core.Solver;
import org.sat4j.minisat.orders.RandomLiteralSelectionStrategy;
import org.sat4j.minisat.orders.RandomWalkDecorator;
import org.sat4j.minisat.orders.VarOrderHeap;
import org.sat4j.reader.DimacsReader;
import org.sat4j.specs.ISolver;
import org.sat4j.specs.IVecInt;
import org.sat4j.specs.TimeoutException;
import org.sat4j.tools.ModelIterator;
import spl.fm.FeaturesPair;
import spl.fm.Product;
import spl.techniques.DistancesUtil;
import spl.techniques.RandomTechnique;
import spl.techniques.SimilarityTechnique;
import spl.techniques.ga.GA;
import spl.techniques.ga.Individual;
import splar.core.fm.FeatureModel;
import splar.core.fm.XMLFeatureModel;
import splar.core.heuristics.FTPreOrderSortedECTraversalHeuristic;
import splar.core.heuristics.VariableOrderingHeuristic;
import splar.core.heuristics.VariableOrderingHeuristicsManager;
import splar.plugins.reasoners.bdd.javabdd.FMReasoningWithBDD;
import splar.plugins.reasoners.bdd.javabdd.ReasoningWithBDD;
import splar.plugins.reasoners.sat.sat4j.FMReasoningWithSAT;
import splar.plugins.reasoners.sat.sat4j.ReasoningWithSAT;

public class SPL {

    private static Random randomGenerator = new Random();
    private FeatureModel fm;
    private ReasoningWithSAT reasonerSAT;
    private ISolver solverIterator, dimacsSolver;
    private final IOrder order = new RandomWalkDecorator(new VarOrderHeap(new RandomLiteralSelectionStrategy()), 1);
    private CommandLineParser parser;
    private static SPL instance = null;
    private final int SATtimeout = 1000;
    private final long iteratorTimeout = 150000;
    private boolean dimacs;
    private String dimacsFile;
    private boolean predictable;

    protected SPL() {
    }

    public static SPL getInstance() {
        if (instance == null) {
            instance = new SPL();
        }
        return instance;
    }

    public final void parseArgs(String[] args) {

        try {
            parser = new CommandLineParser(args, "SPL");
            if (args.length == 0) {
                throw new ParameterException("No arguments");
            }


            parser.parseArgs();
            String command = parser.getCommandName();
            if (command.equals(CommandLineParser.PRIORITIZATION_SOLVER_PRODUCTS)) {
                computePrioritizationSolverProducts(parser.getCommandPrioritizationSolverProducts().fmFile,
                        parser.getCommandPrioritizationSolverProducts().outputFile,
                        parser.getCommandPrioritizationSolverProducts().runs,
                        parser.getCommandPrioritizationSolverProducts().validPairsFile,
                        parser.getCommandPrioritizationSolverProducts().predictable);
            } else if (command.equals(CommandLineParser.AVERAGE_NORMALIZED_DATA_FILES)) {
                if (parser.getCommandAverageDataFiles().outputDirectory == null) {
                    computeAverageDataFiles(parser.getCommandAverageDataFiles().inputDirectory,
                            parser.getCommandAverageDataFiles().inputDirectory,
                            parser.getCommandAverageDataFiles().noNorm);
                } else {
                    computeAverageDataFiles(parser.getCommandAverageDataFiles().inputDirectory,
                            parser.getCommandAverageDataFiles().outputDirectory,
                            parser.getCommandAverageDataFiles().noNorm);
                }
            } else if (command.equals(CommandLineParser.PRIORITIZATION_SPLCAT_PRODUCTS)) {
                computePrioritizationSPLCATProducts(parser.getCommandPrioritizationSPLCATProducts().csvFile,
                        parser.getCommandPrioritizationSPLCATProducts().fmFile,
                        parser.getCommandPrioritizationSPLCATProducts().outputFile,
                        parser.getCommandPrioritizationSPLCATProducts().runs,
                        parser.getCommandPrioritizationSPLCATProducts().validPairsFile);
            } else if (command.equals(CommandLineParser.PRIORITIZATION_SPLCAT_SOLVER_PRODUCTS)) {
                computePrioritizationSPLCATSolverProducts(parser.getCommandPrioritizationSPLCATAndSolverProducts().csvFile,
                        parser.getCommandPrioritizationSPLCATAndSolverProducts().fmFile,
                        parser.getCommandPrioritizationSPLCATAndSolverProducts().outputFile,
                        parser.getCommandPrioritizationSPLCATAndSolverProducts().runs,
                        parser.getCommandPrioritizationSPLCATAndSolverProducts().validPairsFile,
                        parser.commandPrioritizationSPLCATSolverProducts.predictable);
            } else if (command.equals(CommandLineParser.GENERATE_GA)) {
                generateProductsWithGA(parser.getCommandGenerateGA().fmFile,
                        parser.getCommandGenerateGA().csvFile,
                        parser.getCommandGenerateGA().outputFile,
                        parser.getCommandGenerateGA().nbProds,
                        //                        parser.getCommandGenerateGA().popSize,
                        parser.getCommandGenerateGA().runs,
                        parser.getCommandGenerateGA().timeAllowed,
                        parser.getCommandGenerateGA().validPairsFile,
                        parser.getCommandGenerateGA().dimacs,
                        parser.getCommandGenerateGA().noCoverage,
                        parser.getCommandGenerateGA().onlyGA);
            } else if (command.equals(CommandLineParser.NORMALIZE_DATA_FILES)) {
                normalizeDataFile(parser.getCommandNormalizeDataFiles().input);
            } else if (command.equals(CommandLineParser.COMPUTE_PAIRS)) {
                computeValidPairsToFile(parser.getCommandComputePairs().fmFile,
                        parser.getCommandComputePairs().dimacs,
                        parser.getCommandComputePairs().nbParts,
                        parser.getCommandComputePairs().part);
            } else if (command.endsWith(CommandLineParser.COMPUTE_STATS)) {
                computeStats(parser.getCommandComputeStats().fmFile,
                        parser.getCommandComputeStats().dimacs);
            }
        } catch (ParameterException ex) {
            System.out.println("ERROR: " + ex.getMessage());
            parser.printUsage();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void computePrioritizationSPLCATProducts(String splcatFile, String fmFile, String outputDir, int runs, String validPairsFile) throws Exception {
        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }
        File splcatCSV = new File(splcatFile);
        if (!splcatCSV.exists()) {
            throw new ParameterException("The specified SPLCAT file does not exist");
        }
        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (validPairsFile != null && !new File(validPairsFile).exists()) {
            throw new ParameterException("The specified valid pairs file does not exist");
        }
        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM (xml) file does not exist");
        }

        fm = loadFeatureModel(fmFile);
        fm.loadModel();
        reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
        reasonerSAT.init();

        String[] features = reasonerSAT.getVarIndex2NameMap();
        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
        computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);

        String splcatFileName = splcatCSV.getName();

        List<Product> splcatProducts = loadProductsFromCSVFile(splcatCSV, featuresMapRev);
        int productsSize = splcatProducts.size();

        Set<FeaturesPair> validPairs;
        if (validPairsFile != null) {
            validPairs = loadPairs(validPairsFile);
        } else {
            validPairs = computePairsCoveredByProducts(splcatProducts);
        }


        SimilarityTechnique simJaccardGreedy = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.GREEDY_SEARCH);
        SimilarityTechnique simJaccardOpti = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH);
        RandomTechnique random = new RandomTechnique();

        double[] coverageValuesOriginal = new double[productsSize];
        double[] fitnessValuesOriginal;
        double[] coverageValuesRand = new double[productsSize];
        double[] fitnessValuesRand = new double[productsSize];
        double[] coverageValuesSimGreedy = new double[productsSize];
        double[] fitnessValuesSimGreedy;
        double[] coverageValuesSimOpti = new double[productsSize];
        double[] fitnessValuesSimOpti;

        // Original order
        computeProductsCoverage(splcatProducts, validPairs);
        for (int j = 0; j < productsSize; j++) {
            coverageValuesOriginal[j] += splcatProducts.get(j).getCoverage();
        }

        fitnessValuesOriginal = computeFitnessSums(splcatProducts, SimilarityTechnique.JACCARD_DISTANCE);
        shuffle(splcatProducts);

        // Random
        for (int i = 0; i < runs; i++) {


            List<Product> randomProducts = random.prioritize(splcatProducts);
            computeProductsCoverage(randomProducts, validPairs);
            for (int j = 0; j < randomProducts.size(); j++) {
                coverageValuesRand[j] += randomProducts.get(j).getCoverage();
            }
            double[] fitnessRand = computeFitnessSums(randomProducts, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessRand.length; j++) {
                fitnessValuesRand[j] += fitnessRand[j];

            }
        }

        for (int i = 0; i < coverageValuesRand.length; i++) {
            coverageValuesRand[i] /= runs;
            fitnessValuesRand[i] /= runs;
        }

        // Similarity (Greedy)
        List<Product> simGreedyPrioritized = simJaccardGreedy.prioritize(splcatProducts);
        computeProductsCoverage(simGreedyPrioritized, validPairs);
        for (int j = 0; j < simGreedyPrioritized.size(); j++) {
            coverageValuesSimGreedy[j] += simGreedyPrioritized.get(j).getCoverage();
        }
        fitnessValuesSimGreedy = computeFitnessSums(simGreedyPrioritized, SimilarityTechnique.JACCARD_DISTANCE);

        // Similarity (Near optimal)
        List<Product> simOptiPrioritized = simJaccardOpti.prioritize(splcatProducts);
        computeProductsCoverage(simOptiPrioritized, validPairs);
        for (int j = 0; j < simOptiPrioritized.size(); j++) {
            coverageValuesSimOpti[j] += simOptiPrioritized.get(j).getCoverage();
        }
        fitnessValuesSimOpti = computeFitnessSums(simOptiPrioritized, SimilarityTechnique.JACCARD_DISTANCE);

        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProducts-OriginalOrder.dat", coverageValuesOriginal, fitnessValuesOriginal);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProducts-Random-" + runs + "runs.dat", coverageValuesRand, fitnessValuesRand);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProducts-SimJaccardGreedy.dat", coverageValuesSimGreedy, fitnessValuesSimGreedy);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProducts-SimJaccardOpti.dat", coverageValuesSimOpti, fitnessValuesSimOpti);

    }

    public static Product getJaccardWorstProduct(List<Product> prods) {
        double dMin = 0.0;
        Product worst = prods.get(0);

        for (int i = 1; i < prods.size(); i++) {
            dMin += DistancesUtil.getJaccardDistance(worst, prods.get(i));
        }

        for (int i = 1; i < prods.size(); i++) {
            double dist = 0.0;
            Product p = prods.get(i);
            for (int j = 0; j < prods.size(); j++) {
                dist += DistancesUtil.getJaccardDistance(p, prods.get(j));
            }
            if (dist < dMin) {
                dMin = dist;
                worst = p;
            }

        }
        return worst;
    }

    public void computeStats(String fmFile, boolean dimacs) throws Exception {
        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }

        if (!dimacs) {
            fm = loadFeatureModel(fmFile);
            fm.loadModel();
            new FTPreOrderSortedECTraversalHeuristic("Pre-CL-MinSpan", fm, FTPreOrderSortedECTraversalHeuristic.FORCE_SORT);
            VariableOrderingHeuristic heuristic = VariableOrderingHeuristicsManager.createHeuristicsManager().getHeuristic("Pre-CL-MinSpan");

            // Creates the BDD reasoner
            ReasoningWithBDD reasonerBDD = new FMReasoningWithBDD(fm, heuristic, 50000, 50000, 60000, "pre-order");

            // Initialize the reasoner (BDD is created at this moment)
            reasonerBDD.init();
            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
            reasonerSAT.init();
            System.out.println("#Configs: " + reasonerBDD.countValidConfigurations());
            System.out.println("#Constraints: " + reasonerSAT.getSolver().nConstraints());
            System.out.println("#Features: " + reasonerSAT.getSolver().nVars());

        } else {
            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
            dimacsSolver.setTimeout(SATtimeout);
            DimacsReader dr = new DimacsReader(dimacsSolver);
            dr.parseInstance(new FileReader(fmFile));
            System.out.println("#Constraints: " + dimacsSolver.nConstraints());
            System.out.println("#Features: " + dimacsSolver.nVars());
        }
    }

    public void generateProductsWithGA(String fmFile, String splcatFile, String outputDir, int nbProds, /*int popSize,*/ int runs, long timeAllowed, String validPairsFile, boolean dimacs, boolean noCoverage, boolean onlyGA) throws Exception {
        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }

        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }

        if (validPairsFile != null && !new File(validPairsFile).exists()) {
            throw new ParameterException("The specified valid pairs file does not exist");
        }

        if (nbProds < 0 && splcatFile == null) {
            throw new ParameterException("If -nbRuns < 0 then the csv file is mandatory!");
        }

        File splcatCSV = null;
        if (splcatFile != null) {
            splcatCSV = new File(splcatFile);
            if (!splcatCSV.exists()) {
                throw new ParameterException("The specified SPLCAT file does not exist");
            }
        }

        this.predictable = false;
        this.dimacs = dimacs;
        this.dimacsFile = fmFile;
        if (!dimacs) {
            fm = loadFeatureModel(fmFile);
            fm.loadModel();
            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
            reasonerSAT.init();
        } else {
            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
            dimacsSolver.setTimeout(SATtimeout);
            DimacsReader dr = new DimacsReader(dimacsSolver);
            dr.parseInstance(new FileReader(fmFile));
        }




        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();

        if (!dimacs) {
            computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);
        } else {
            computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
        }


        System.out.println(featuresMapRev.size() + " features");

        Set<FeaturesPair> validPairs = null;

        if (!noCoverage) {

            if (validPairsFile != null) {
                System.out.println("Loading valid pairs from the file...");
                validPairs = loadPairs(validPairsFile);
            }
        }

        String sNbProds = "" + nbProds;

        if (nbProds < 0) {

            List<Product> splcatProducts = loadProductsFromCSVFile(splcatCSV, featuresMapRev);
            nbProds = splcatProducts.size();
            sNbProds = "SPLCAT";
            if (!noCoverage) {
                if (validPairs == null) {
                    validPairs = computePairsCoveredByProducts(splcatProducts);
                }
            }
        }

        if (!noCoverage) {
            if (validPairs == null) {
                if (!dimacs) {
                    validPairs = computeValidPairs(featuresMap, featuresList, null, false, null, 1, 1);
                } else {
                    computeValidPairs(featuresMap, featuresList, (fmFile + ".validpairs"), true, dimacsSolver, 1, 1);
                }
            }

            System.out.println(validPairs.size() + " valid pairs.");
        }
        System.out.println(nbProds + " products to generate, " + runs + " runs");


        double[] coverageValuesUnpredictable = new double[nbProds];
        double[] fitnessValuesUnpredictable = new double[nbProds];
        double[] coverageValuesUnpredictablePrioritized = new double[nbProds];
        double[] fitnessValuesUnpredictablePrioritized = new double[nbProds];
        double[] coverageValuesSimpleGA = new double[nbProds];
        double[] fitnessValuesSimpleGA = new double[nbProds];


        if (!dimacs) {
            reasonerSAT.init();
            ((Solver) reasonerSAT.getSolver()).setOrder(order);
            solverIterator = new ModelIterator(reasonerSAT.getSolver());
            solverIterator.setTimeoutMs(iteratorTimeout);
        } else {
            ((Solver) dimacsSolver).setOrder(order);
            solverIterator = new ModelIterator(dimacsSolver);
            solverIterator.setTimeoutMs(iteratorTimeout);
        }

        GA ga = new GA(timeAllowed);
        String fmFileName = new File(fmFile).getName();
        System.out.println("Starting the runs...");
        for (int i = 0; i < runs; i++) {
            System.out.println("run " + (i + 1));


            double[] runCoverageUnpredictable = null;
            double[] fitnessUnpredictable = null;
            double[] runCoverageUnpredictablePrioritized = null;
            double[] fitnessUnpredictablePrioritized = null;
            List<Product> unpredictableProducts = null;
            List<Product> simOptiPrioritizedUnpredictable = null;
            if (!onlyGA) {

                //unpredictable products
                System.out.println("Unpredictable...");
                unpredictableProducts = getUnpredictableProducts(nbProds);
                if (!noCoverage) {
                    System.out.println("done, coverage...");
                } else {
                    System.out.println("done.");
                }
                shuffle(unpredictableProducts);



                if (!noCoverage) {

                    computeProductsCoverage(unpredictableProducts, validPairs);
                    runCoverageUnpredictable = new double[coverageValuesUnpredictable.length];

                    for (int j = 0; j < nbProds; j++) {
                        double cov = unpredictableProducts.get(j).getCoverage();
                        coverageValuesUnpredictable[j] += cov;
                        runCoverageUnpredictable[j] = cov;
                    }

                    fitnessUnpredictable = computeFitnessSums(unpredictableProducts, SimilarityTechnique.JACCARD_DISTANCE);
                    for (int j = 0; j < fitnessUnpredictable.length; j++) {
                        fitnessValuesUnpredictable[j] += fitnessUnpredictable[j];

                    }
                }

                //unpredictable prioritized
                System.out.println("unpredictable prioritized...");
                simOptiPrioritizedUnpredictable = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH).prioritize(unpredictableProducts);
                if (!noCoverage) {
                    System.out.println("done, coverage...");
                } else {
                    System.out.println("done.");
                }


                if (!noCoverage) {

                    computeProductsCoverage(simOptiPrioritizedUnpredictable, validPairs);
                    runCoverageUnpredictablePrioritized = new double[coverageValuesUnpredictablePrioritized.length];
                    for (int j = 0; j < nbProds; j++) {
                        double cov = simOptiPrioritizedUnpredictable.get(j).getCoverage();
                        coverageValuesUnpredictablePrioritized[j] += cov;
                        runCoverageUnpredictablePrioritized[j] = cov;
                    }
                    fitnessUnpredictablePrioritized = computeFitnessSums(simOptiPrioritizedUnpredictable, SimilarityTechnique.JACCARD_DISTANCE);
                    for (int j = 0; j < fitnessValuesUnpredictablePrioritized.length; j++) {
                        fitnessValuesUnpredictablePrioritized[j] += fitnessUnpredictablePrioritized[j];

                    }
                }

            }

            System.out.println("Simple GA...");
            List<Product> gaSimpleRes = ga.runSimpleGA(nbProds, Individual.MUTATE_WORST).getProducts();
            if (!noCoverage) {
                System.out.println("done, coverage...");
            } else {
                System.out.println("done.");
            }


            double[] runCoverageGA = null;
            double[] fitnessSimpleGA = null;
            if (!noCoverage) {
                computeProductsCoverage(gaSimpleRes, validPairs);
                runCoverageGA = new double[coverageValuesSimpleGA.length];
                for (int j = 0; j < nbProds; j++) {
                    double cov = gaSimpleRes.get(j).getCoverage();
                    coverageValuesSimpleGA[j] += cov;
                    runCoverageGA[j] = cov;
                }
                fitnessSimpleGA = computeFitnessSums(gaSimpleRes, SimilarityTechnique.JACCARD_DISTANCE);
                for (int j = 0; j < fitnessValuesSimpleGA.length; j++) {
                    fitnessValuesSimpleGA[j] += fitnessSimpleGA[j];

                }
            }

            //run values
            if (!noCoverage) {
                if (!onlyGA) {
                    writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProducts-" + sNbProds + "prods-" + timeAllowed + "ms-run" + (i + 1) + ".dat", runCoverageUnpredictable, fitnessUnpredictable);
                    writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProductsPrioritized-" + sNbProds + "prods-" + timeAllowed + "ms-run" + (i + 1) + ".dat", runCoverageUnpredictablePrioritized, fitnessUnpredictablePrioritized);
                }
                writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-SimpleGAProducts-" + sNbProds + "prods-" + timeAllowed + "ms-run" + (i + 1) + ".dat", runCoverageGA, fitnessSimpleGA);
            }
            //save products

            if (!onlyGA) {
                writeProductsToFile(outputDir + fmFileName + "_GA-UnpredictableProducts-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.csv", unpredictableProducts, featuresMap, featuresList);
                writeProductsToFile(outputDir + fmFileName + "_GA-UnpredictablePrioritized-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.csv", simOptiPrioritizedUnpredictable, featuresMap, featuresList);
            }
            writeProductsToFile(outputDir + fmFileName + "_GA-SimpleGAProducts-" + sNbProds + "prods-" + timeAllowed + "ms-" + "run" + (i + 1) + ".products.csv", gaSimpleRes, featuresMap, featuresList);
        }

        if (!noCoverage) {
            for (int i = 0; i < nbProds; i++) {
                if (!onlyGA) {
                    coverageValuesUnpredictable[i] /= runs;
                    coverageValuesUnpredictablePrioritized[i] /= runs;

                    fitnessValuesUnpredictable[i] /= runs;
                    fitnessValuesUnpredictablePrioritized[i] /= runs;
                }
                coverageValuesSimpleGA[i] /= runs;
                fitnessValuesSimpleGA[i] /= runs;
            }
        }

        if (!noCoverage) {
            if (!onlyGA) {
                writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProducts-" + sNbProds + "prods-" + timeAllowed + "ms-" + runs + "runs.dat", coverageValuesUnpredictable, fitnessValuesUnpredictable);
            }
            writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-UnpredictableProductsPrioritized-" + sNbProds + "prods-" + timeAllowed + "ms-" + runs + "runs.dat", coverageValuesUnpredictablePrioritized, fitnessValuesUnpredictablePrioritized);
            writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_GA-SimpleGAProducts-" + sNbProds + "prods-" + timeAllowed + "ms-" + runs + "runs.dat", coverageValuesSimpleGA, fitnessValuesSimpleGA);
        }


    }

    public void computePrioritizationSPLCATSolverProducts(String splcatFile, String fmFile, String outputDir, int runs, String validPairsFile, boolean predictable) throws Exception {
        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }
        File splcatCSV = new File(splcatFile);
        if (!splcatCSV.exists()) {
            throw new ParameterException("The specified SPLCAT file does not exist");
        }
        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM (xml) file does not exist");
        }

        if (validPairsFile != null && !new File(validPairsFile).exists()) {
            throw new ParameterException("The specified valid pairs file does not exist");
        }

        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        fm = loadFeatureModel(fmFile);
        fm.loadModel();
        reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
        reasonerSAT.init();

        this.predictable = predictable;


        String[] features = reasonerSAT.getVarIndex2NameMap();
        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
        computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);



        String splcatFileName = splcatCSV.getName();

        List<Product> splcatProducts = loadProductsFromCSVFile(splcatCSV, featuresMapRev);
        int splcatProductsSize = splcatProducts.size();
        int productsSize = splcatProducts.size() * 2;

        SimilarityTechnique simJaccardGreedy = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.GREEDY_SEARCH);
        SimilarityTechnique simJaccardOpti = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH);
        RandomTechnique random = new RandomTechnique();

        double[] coverageValuesSolver = new double[productsSize];
        double[] fitnessValuesSolver = new double[productsSize];
        double[] coverageValuesSimGreedy = new double[productsSize];
        double[] fitnessValuesSimGreedy = new double[productsSize];
        double[] coverageValuesSimOpti = new double[productsSize];
        double[] fitnessValuesSimOpti = new double[productsSize];

        shuffle(splcatProducts);
        reasonerSAT.init();
        if (!predictable) {
            ((Solver) reasonerSAT.getSolver()).setOrder(order);
        }
        solverIterator = new ModelIterator(reasonerSAT.getSolver());
        solverIterator.setTimeoutMs(iteratorTimeout);

        Set<FeaturesPair> validPairs;

        if (validPairsFile != null) {
            validPairs = loadPairs(validPairsFile);
        } else {
            validPairs = computePairsCoveredByProducts(splcatProducts);
        }
        for (int i = 0; i < runs; i++) {

            // SPLCAT + solver products

            List<Product> solverProducts;
            if (!predictable) {
                solverProducts = getUnpredictableProducts(splcatProductsSize);
            } else {
                solverProducts = getPredictableProducts(splcatProductsSize, features.length);
            }
            List<Product> allProducts = new ArrayList<Product>(splcatProducts);
            allProducts.addAll(solverProducts);

            shuffle(allProducts);


            // Solver
            List<Product> solverPrioritized = random.prioritize(allProducts);
            computeProductsCoverage(solverPrioritized, validPairs);
            for (int j = 0; j < solverPrioritized.size(); j++) {
                coverageValuesSolver[j] += solverPrioritized.get(j).getCoverage();
            }

            double[] fitnessSolver = computeFitnessSums(solverPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSolver.length; j++) {
                fitnessValuesSolver[j] += fitnessSolver[j];

            }

            // Similarity (Greedy)
            List<Product> simGreedyPrioritized = simJaccardGreedy.prioritize(allProducts);
            computeProductsCoverage(simGreedyPrioritized, validPairs);
            for (int j = 0; j < simGreedyPrioritized.size(); j++) {
                coverageValuesSimGreedy[j] += simGreedyPrioritized.get(j).getCoverage();
            }
            double[] fitnessSimGreedy = computeFitnessSums(simGreedyPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSimGreedy.length; j++) {
                fitnessValuesSimGreedy[j] += fitnessSimGreedy[j];

            }

            // Similarity (Near optimal)
            List<Product> simOptiPrioritized = simJaccardOpti.prioritize(allProducts);
            computeProductsCoverage(simOptiPrioritized, validPairs);
            for (int j = 0; j < simOptiPrioritized.size(); j++) {
                coverageValuesSimOpti[j] += simOptiPrioritized.get(j).getCoverage();
            }
            double[] fitnessSimOpti = computeFitnessSums(simOptiPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSimOpti.length; j++) {
                fitnessValuesSimOpti[j] += fitnessSimOpti[j];

            }

        }
        for (int i = 0; i < coverageValuesSolver.length; i++) {
            coverageValuesSolver[i] /= runs;
            coverageValuesSimGreedy[i] /= runs;
            coverageValuesSimOpti[i] /= runs;
            fitnessValuesSolver[i] /= runs;
            fitnessValuesSimGreedy[i] /= runs;
            fitnessValuesSimOpti[i] /= runs;
        }

        String sSolver = predictable ? "Predictable-" : "Unpredictable-";
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProductsAndSolver-" + sSolver + runs + "runs.dat", coverageValuesSolver, fitnessValuesSolver);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProductsAndSolver-SimJaccardGreedy-" + runs + "runs.dat", coverageValuesSimGreedy, fitnessValuesSimGreedy);
        writeCoverageAndFitnessValuesToFile(outputDir + splcatFileName + "_SPLCATProductsAndSolver-SimJaccardOpti-" + runs + "runs.dat", coverageValuesSimOpti, fitnessValuesSimOpti);

    }

    public void computePrioritizationSolverProducts(String fmFile, String outputDir, int runs, String validPairsFile, boolean predictable) throws Exception {

        File output = new File(outputDir);
        if (!output.isDirectory()) {
            throw new ParameterException("Output directory specified is incorrect");
        }
        if (runs < 1) {
            throw new ParameterException("Number of runs specified incorrect");
        }

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }
        if (!outputDir.endsWith("/")) {
            outputDir += "/";
        }

        if (validPairsFile != null && !new File(validPairsFile).exists()) {
            throw new ParameterException("The specified valid pairs file does not exist");
        }

        String fmFileName = new File(fmFile).getName();
        fm = loadFeatureModel(fmFile);
        fm.loadModel();
        reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, SATtimeout);
        reasonerSAT.init();

        String[] features = reasonerSAT.getVarIndex2NameMap();
        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
        computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);

        reasonerSAT.init();


        this.predictable = predictable;

        Set<FeaturesPair> validPairs;
        if (validPairsFile != null) {
            validPairs = loadPairs(validPairsFile);
        } else {
            validPairs = computeValidPairs(featuresMap, featuresList, null, false, null, 1, 1);
        }
        int featuresCount = features.length;
        int productsSize = featuresCount / 2;

        double[] coverageValuesSolver = new double[productsSize];
        double[] fitnessValuesSolver = new double[productsSize];
        double[] coverageValuesSimGreedy = new double[productsSize];
        double[] fitnessValuesSimGreedy = new double[productsSize];
        double[] coverageValuesSimOpti = new double[productsSize];
        double[] fitnessValuesSimOpti = new double[productsSize];

        SimilarityTechnique simJaccardGreedy = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.GREEDY_SEARCH);
        SimilarityTechnique simJaccardOpti = new SimilarityTechnique(SimilarityTechnique.JACCARD_DISTANCE, SimilarityTechnique.NEAR_OPTIMAL_SEARCH);

        reasonerSAT.init();
        if (!predictable) {
            ((Solver) reasonerSAT.getSolver()).setOrder(order);
        }
        solverIterator = new ModelIterator(reasonerSAT.getSolver());
        solverIterator.setTimeoutMs(iteratorTimeout);

        for (int i = 0; i < runs; i++) {

            // Solver
            List<Product> solverProducts;
            if (!predictable) {
                solverProducts = getUnpredictableProducts(productsSize);
            } else {
                solverProducts = getPredictableProducts(productsSize, features.length);
            }


            shuffle(solverProducts);
            computeProductsCoverage(solverProducts, validPairs);
            for (int j = 0; j < solverProducts.size(); j++) {
                coverageValuesSolver[j] += solverProducts.get(j).getCoverage();
            }
            double[] fitnessSolver = computeFitnessSums(solverProducts, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSolver.length; j++) {
                fitnessValuesSolver[j] += fitnessSolver[j];

            }

            // Similarity (Greedy)
            List<Product> simGreedyPrioritized = simJaccardGreedy.prioritize(solverProducts);
            // The pairs covered are the same
            computeProductsCoverage(simGreedyPrioritized, validPairs);

            for (int j = 0; j < simGreedyPrioritized.size(); j++) {
                coverageValuesSimGreedy[j] += simGreedyPrioritized.get(j).getCoverage();
            }
            double[] fitnessSimGreedy = computeFitnessSums(simGreedyPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSimGreedy.length; j++) {
                fitnessValuesSimGreedy[j] += fitnessSimGreedy[j];

            }

            // Similarity (Near optimal)
            List<Product> simOptiPrioritized = simJaccardOpti.prioritize(solverProducts);
            // The pairs covered are the same
            computeProductsCoverage(simOptiPrioritized, validPairs);
            for (int j = 0; j < simOptiPrioritized.size(); j++) {
                coverageValuesSimOpti[j] += simOptiPrioritized.get(j).getCoverage();
            }
            double[] fitnessSimOpti = computeFitnessSums(simOptiPrioritized, SimilarityTechnique.JACCARD_DISTANCE);
            for (int j = 0; j < fitnessSimOpti.length; j++) {
                fitnessValuesSimOpti[j] += fitnessSimOpti[j];

            }
        }

        for (int i = 0; i < coverageValuesSolver.length; i++) {
            coverageValuesSolver[i] /= runs;
            coverageValuesSimGreedy[i] /= runs;
            coverageValuesSimOpti[i] /= runs;
            fitnessValuesSolver[i] /= runs;
            fitnessValuesSimGreedy[i] /= runs;
            fitnessValuesSimOpti[i] /= runs;
        }
        String sSolver = predictable ? "Predictable-" : "Unpredictable-";
        writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_ProductsSolver-" + sSolver + runs + "runs.dat", coverageValuesSolver, fitnessValuesSolver);
        writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_ProductsSolver-SimJaccardGreedy-" + runs + "runs.dat", coverageValuesSimGreedy, fitnessValuesSimGreedy);
        writeCoverageAndFitnessValuesToFile(outputDir + fmFileName + "_ProductsSolver-SimJaccardOpti-" + runs + "runs.dat", coverageValuesSimOpti, fitnessValuesSimOpti);
    }

    public FeatureModel loadFeatureModel(String fmFile) {
        return new XMLFeatureModel(fmFile, XMLFeatureModel.USE_VARIABLE_NAME_AS_ID);
    }

    public List<Product> getUnpredictableProducts(int count) throws Exception {
        List<Product> products = new ArrayList<Product>(count);

        while (products.size() < count) {

            try {
                if (solverIterator.isSatisfiable()) {
                    Product product = toProduct(solverIterator.model());

                    if (!products.contains(product)) {
                        products.add(product);
                    }

                } else {
                    if (!dimacs) {
                        reasonerSAT.init();
                        if (!predictable) {
                            ((Solver) reasonerSAT.getSolver()).setOrder(order);
                        }
                        solverIterator = new ModelIterator(reasonerSAT.getSolver());
                        solverIterator.setTimeoutMs(iteratorTimeout);

                    } else {
                        dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                        dimacsSolver.setTimeout(SATtimeout);
                        DimacsReader dr = new DimacsReader(dimacsSolver);
                        dr.parseInstance(new FileReader(dimacsFile));
                        if (!predictable) {
                            ((Solver) dimacsSolver).setOrder(order);
                        }
                        solverIterator = new ModelIterator(dimacsSolver);
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    }
                }
            } catch (TimeoutException e) {
            }
        }
        return products;
    }

    public Product getUnpredictableProduct() throws Exception {
        Product product = null;
        while (product == null) {
            try {
                if (solverIterator.isSatisfiable()) {
                    product = toProduct(solverIterator.model());
                } else {
                    if (!dimacs) {
                        reasonerSAT.init();
                        if (!predictable) {
                            ((Solver) reasonerSAT.getSolver()).setOrder(order);
                        }
                        solverIterator = new ModelIterator(reasonerSAT.getSolver());
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    } else {
                        dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                        dimacsSolver.setTimeout(SATtimeout);
                        DimacsReader dr = new DimacsReader(dimacsSolver);
                        dr.parseInstance(new FileReader(dimacsFile));
                        if (!predictable) {
                            ((Solver) dimacsSolver).setOrder(order);
                        }
                        solverIterator = new ModelIterator(dimacsSolver);
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    }
                }
            } catch (TimeoutException e) {
            }
        }
        return product;
    }

    public List<Product> getPredictableProducts(int count, int numberOfFeatures) throws Exception {
        List<Product> products = new ArrayList<Product>(count);
        while (products.size() < count) {
            try {
                if (solverIterator.isSatisfiable()) {
                    Product product = toProduct(solverIterator.model());
                    if (randomGenerator.nextInt(numberOfFeatures) == numberOfFeatures - 1) {

                        if (!products.contains(product)) {
                            products.add(product);
                        }
                    }
                } else {
                    if (!dimacs) {
                        reasonerSAT.init();
                        if (!predictable) {
                            ((Solver) reasonerSAT.getSolver()).setOrder(order);
                        }
                        solverIterator = new ModelIterator(reasonerSAT.getSolver());
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    } else {
                        dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
                        dimacsSolver.setTimeout(SATtimeout);
                        DimacsReader dr = new DimacsReader(dimacsSolver);
                        dr.parseInstance(new FileReader(dimacsFile));
                        if (!predictable) {
                            ((Solver) dimacsSolver).setOrder(order);
                        }
                        solverIterator = new ModelIterator(dimacsSolver);
                        solverIterator.setTimeoutMs(iteratorTimeout);
                    }
                }

            } catch (TimeoutException e) {
            }
        }
        return products;
    }

    public Set<FeaturesPair> computePairsCoveredByProducts(List<Product> products) {
        Set<FeaturesPair> coveredPairs = new HashSet<FeaturesPair>();
        for (Product product : products) {
            Set<FeaturesPair> productPairs = product.computeCoveredPairs();
            coveredPairs.addAll(productPairs);
        }
        return coveredPairs;
    }

    public void computeProductsCoverage(List<Product> products, Set<FeaturesPair> pairs) {
        double pairsSize = pairs.size();
        Set<FeaturesPair> pairsCopy = new HashSet<FeaturesPair>(pairs);
        int n = 1;
        for (Product product : products) {
            int initialSize = pairsCopy.size();
            Set<FeaturesPair> productPairs = product.computeCoveredPairs();
            pairsCopy.removeAll(productPairs);
            double removed = initialSize - pairsCopy.size();
            double coverage = removed / pairsSize * 100.0;
            product.setCoverage(coverage);
        }
        pairsCopy = null;
    }

    public void shuffle(List<Product> products) {
        List<Product> productsCopy = new ArrayList<Product>(products);
        int done = 0;
        while (done < products.size()) {
            int index = randomGenerator.nextInt(productsCopy.size());
            products.set(done++, productsCopy.get(index));
            productsCopy.remove(index);
        }
    }

    public void writeDimacsProductToFile(String outFile, Product product) throws Exception {
        BufferedWriter out = new BufferedWriter(new FileWriter(outFile));

        for (Integer i : product) {
            out.write(Integer.toString(i));
            //if (n < product.size()) {
            out.newLine();
            //}
        }
        out.close();
    }

    public void writeProductsToFile(String outFile, List<Product> products, Map<Integer, String> featuresMap, List<Integer> featuresList) throws Exception {
        BufferedWriter out = new BufferedWriter(new FileWriter(outFile));

        out.write("Feature\\Product;");

        for (int i = 0; i < products.size(); i++) {
            out.write("" + i + ";");
        }

        out.newLine();

        int featuresCount = featuresList.size() / 2;
        for (int i = 1; i <= featuresCount; i++) {
            out.write(featuresMap.get(i) + ";");

            for (Product p : products) {
                for (Integer n : p) {
                    if (Math.abs(n) == i) {
                        if (n > 0) {
                            out.write("X;");
                        } else {
                            out.write("-;");
                        }
                    }
                }
            }
            out.newLine();
        }
        out.close();


        //        BufferedWriter out = new BufferedWriter(new FileWriter(outFile));
//
//        int featuresCount = featuresList.size() / 2;
//        for (int i = 1; i <= featuresCount; i++) {
//            out.write(i + ":" + featuresMap.get(i));
//            if (i < featuresCount) {
//                out.write(";");
//            }
//        }
//        out.newLine();
//        for (Product product : products) {
//            List<Integer> prodFeaturesList = new ArrayList<Integer>(product);
//            Collections.sort(prodFeaturesList, new Comparator<Integer>() {
//
//                @Override
//                public int compare(Integer o1, Integer o2) {
//                    return ((Integer) Math.abs(o1)).compareTo(((Integer) Math.abs(o2)));
//                }
//            });
//
//            int done = 1;
//            for (Integer feature : prodFeaturesList) {
//                out.write((feature > 0 ? "X" : "-"));
//                if (done < featuresCount) {
//                    out.write(";");
//                }
//                done++;
//            }
//
//            out.newLine();
//        }
//        out.close();
    }

    public boolean isValidProduct(Product product, Map<Integer, String> featuresMap, List<Integer> featuresList) throws Exception {
        IVecInt prod = new VecInt(product.size());

        for (Integer s : product) {

            if (s < 0) {
                prod.push(-reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get((-s) - 1))));
            } else {
                prod.push(reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get(s - 1))));
            }
        }
        return reasonerSAT.getSolver().isSatisfiable(prod);
    }

    public boolean isValidPair(FeaturesPair pair, Map<Integer, String> featuresMap, List<Integer> featuresList, boolean dimacs, ISolver dimacsSolver) throws Exception {

        IVecInt prod = new VecInt(2);
        int f1 = pair.getX();
        int f2 = pair.getY();

        if (!dimacs) {

            if (f1 < 0) {
                prod.push(-reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get((-f1) - 1))));
            } else {
                prod.push(reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get(f1 - 1))));
            }

            if (f2 < 0) {
                prod.push(-reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get((-f2) - 1))));
            } else {
                prod.push(reasonerSAT.getVariableIndex(featuresMap.get(featuresList.get(f2 - 1))));
            }

            return reasonerSAT.getSolver().isSatisfiable(prod);
        } else {
            prod.push(f1);
            prod.push(f2);

            return dimacsSolver.isSatisfiable(prod);
        }
    }

    private Set<FeaturesPair> computeValidPairs(Map<Integer, String> featuresMap, List<Integer> featuresList,
            String outFile, boolean dimacs, ISolver dimacsSolver, int nbParts, int part) throws Exception {


        if (part > nbParts || nbParts < 1) {
            throw new ParameterException("Invalid parts parameters");
        }


        BufferedWriter out = null;
        if (outFile != null) {

            if (nbParts == 1) {
                out = new BufferedWriter(new FileWriter(outFile));
            } else {
                out = new BufferedWriter(new FileWriter(outFile + ".part" + part));
            }
        }

        Set<FeaturesPair> pairs = null;

        if (outFile == null) {
            pairs = new HashSet<FeaturesPair>();
            pairs.clear();
        }




        int size = featuresList.size();

        if (nbParts != 1) {

            int combSize = size * (size - 1) / 2;
            int tick = combSize / nbParts;
            int iStart = 0;

            if (part == 1) {
                iStart = 0;
            } else {
                for (int i = 1; i < part; i++) {
                    iStart += tick;
                }
            }

            int iEnd = 0;
            if (part == nbParts) {
                iEnd = combSize - 1;
            } else {
                iEnd = iStart + tick - 1;
            }

            int done = 0;
            int n = 0;
            long start = System.currentTimeMillis();
            for (int i = 0; i < size; i++) {
                for (int j = i + 1; j < size; j++) {
                    if (System.currentTimeMillis() - start > 30000) {
                        System.out.println(done + "/" + tick);
                        start = System.currentTimeMillis();
                    }

                    if (n >= iStart && n <= iEnd) {
                        done++;
                        int left = featuresList.get(i);
                        int right = featuresList.get(j);
                        if (Math.abs(left) != Math.abs(right)) {
                            FeaturesPair pair = new FeaturesPair(left, right);
                            boolean validPair;
                            if (dimacs) {
                                validPair = isValidPair(pair, null, null, true, dimacsSolver);
                            } else {
                                validPair = isValidPair(pair, featuresMap, featuresList, false, null);
                            }

                            if (validPair) {
                                if (outFile != null) {
                                    out.write(pair.getX() + ";" + pair.getY());
                                    out.newLine();
                                } else {
                                    pairs.add(pair);
                                }
                            }
                        }
                    }
                    n++;
                }
            }
        } else {

            for (int i = 0; i < size; i++) {

                int left = featuresList.get(i);
                for (int j = i + 1; j < size; j++) {

                    int right = featuresList.get(j);
                    if (Math.abs(left) != Math.abs(right)) {
                        FeaturesPair pair = new FeaturesPair(left, right);

                        boolean validPair;
                        if (dimacs) {
                            validPair = isValidPair(pair, null, null, true, dimacsSolver);
                        } else {
                            validPair = isValidPair(pair, featuresMap, featuresList, false, null);
                        }

                        if (validPair) {
                            if (outFile != null) {
                                out.write(pair.getX() + ";" + pair.getY());
                                out.newLine();
                            } else {
                                pairs.add(pair);
                            }
                        }
                    }
                }
            }
        }

        if (outFile != null) {
            out.close();
            return null;
        }
        return pairs;
    }

    public void computeFeatures(ReasoningWithSAT reasonerSAT, Map<Integer, String> featuresMap, Map<String, Integer> featuresMapRev, List<Integer> featuresList, boolean dimacs, String dimacsFile) throws Exception {

        if (!dimacs) {
            String[] features = reasonerSAT.getVarIndex2NameMap();

            for (int i = 0; i < features.length; i++) {
                String feature = features[i];
                int n = i + 1;
                featuresList.add(n);
                featuresMap.put(n, feature);
                featuresMapRev.put(feature, n);
            }


        } else {
            BufferedReader in = new BufferedReader(new FileReader(dimacsFile));
            String line;
            int n = 0;
            while ((line = in.readLine()) != null && line.startsWith("c")) {
                StringTokenizer st = new StringTokenizer(line.trim(), " ");
                st.nextToken();
                n++;
                String sFeature = st.nextToken().replace('$', ' ').trim();
                int feature = Integer.parseInt(sFeature);
                if (n != feature) {
                    throw new Exception("Incorrect dimacs file, missing feature number " + n + " ?");
                }
                String featureName = st.nextToken();
                featuresList.add(feature);
                featuresMap.put(feature, featureName);
                featuresMapRev.put(featureName, feature);
            }
            in.close();
        }

        int n = 1;
        int featuresCount = featuresList.size();
        while (n <= featuresCount) {
            featuresList.add(-n);
            n++;
        }


    }

    public void writeCoverageAndFitnessValuesToFile(String outFile, double[] coverageValues, double[] fitnessSums) throws IOException {
        BufferedWriter out = new BufferedWriter(new FileWriter(outFile));
        out.write("#coverage of pairs (in %, starting from 0 products selected);Fitness func");
        out.newLine();
        out.write("0;0");
        out.newLine();
        double s = 0;
        for (int i = 0; i < coverageValues.length; i++) {
            s += coverageValues[i];
            out.write(Double.toString(s) + ";" + Double.toString(fitnessSums[i]));
            out.newLine();
        }
        out.close();
    }

    public void normalizeDataFile(String inputDir) throws Exception {

        File inDir = new File(inputDir);
        if (!inDir.exists()) {
            throw new ParameterException("Input directory does not exist");
        }

        File[] datFiles = inDir.listFiles(new FilenameFilter() {

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".dat") && !name.toLowerCase().contains("norm");
            }
        });

        for (File file : datFiles) {

            int count = countUncommentedLines(file);

            double[] coverageValues = new double[count];
            double[] fitnessValues = new double[count];

            BufferedReader in = new BufferedReader(new FileReader(file));

            int i = 0;
            String line;

            while ((line = in.readLine()) != null) {
                line = line.trim();
                if (!line.startsWith("#")) {
                    StringTokenizer st = new StringTokenizer(line, ";");

                    coverageValues[i] = Double.parseDouble(st.nextToken().trim());
                    fitnessValues[i] = Double.parseDouble(st.nextToken().trim());
                    i++;
                }
            }
            in.close();

            double[] normalizedCoverageValues = new double[101];
            double[] normalizedFitnessValues = new double[101];

            for (int j = 0; j < normalizedCoverageValues.length; j++) {
                int prodIndex = (int) ((double) j / 100.0 * (coverageValues.length - 1));
                normalizedCoverageValues[j] = coverageValues[prodIndex];
                normalizedFitnessValues[j] = fitnessValues[prodIndex] / fitnessValues[fitnessValues.length - 1] * 100;
            }


            String outFile = file.toString().replace(".dat", "-Norm.dat");
            BufferedWriter out = new BufferedWriter(new FileWriter(outFile));
            out.write("#coverage of pairs (in %, starting from 0% of products selected (normalized));Fitness func (normalized)");
            out.newLine();
            for (int k = 0; k < normalizedCoverageValues.length; k++) {
                out.write(Double.toString(normalizedCoverageValues[k]) + ";" + Double.toString(normalizedFitnessValues[k]));
                out.newLine();
            }
            out.close();
        }

    }

    public Product toProduct(int[] vector) {

        Product product = new Product();
        for (int i : vector) {
            product.add(i);
        }
        return product;
    }

    public void computeAverageDataFiles(String inputDir, String outputDir, final boolean noNorm) throws Exception {
        File inDir = new File(inputDir);
        if (!inDir.exists()) {
            throw new ParameterException("Input directory does not exist");
        }

        if (outputDir.equals("Same as input")) {
            outputDir = inputDir;
        }

        if (!new File(outputDir).exists()) {
            throw new ParameterException("Output directory does not exist");
        }
        File[] datFiles = inDir.listFiles(new FilenameFilter() {

            @Override
            public boolean accept(File dir, String name) {
                if (!noNorm) {
                    return name.endsWith("-Norm.dat");
                } else {
                    return name.endsWith(".dat") && !name.toLowerCase().contains("norm");
                }
            }
        });

        Set<String> types = new HashSet<String>();
        for (File file : datFiles) {
            String sFile = file.toString();
            String type = sFile.substring(sFile.lastIndexOf("_") + 1, sFile.length());
            types.add(type);
        }
        for (final String type : types) {
            datFiles = inDir.listFiles(new FilenameFilter() {

                @Override
                public boolean accept(File dir, String name) {
                    return name.endsWith(type);
                }
            });
            int n = 0;
            double[] coverageValues, fitnessValues;
            if (!noNorm) {
                coverageValues = new double[101];
                fitnessValues = new double[101];
            } else {
                int count = minUncommentedLinesCount(datFiles);
                coverageValues = new double[count];
                fitnessValues = new double[count];
            }

            String firstLine = "";
            for (File dat : datFiles) {
                int i = 0;
                BufferedReader in = new BufferedReader(new FileReader(dat));
                String line;
                while ((line = in.readLine()) != null && i < coverageValues.length) {
                    line = line.trim();
                    if (!line.isEmpty()) {
                        if (line.startsWith("#")) {
                            firstLine = line;
                        } else {
                            StringTokenizer tokenizer = new StringTokenizer(line, ";");
                            double cov = Double.parseDouble(tokenizer.nextToken());
                            double fit = Double.parseDouble(tokenizer.nextToken());
                            coverageValues[i] += cov;
                            fitnessValues[i] += fit;
                            i++;
                        }
                    }
                }
                in.close();
                n++;

            }

            for (int i = 0; i < coverageValues.length; i++) {
                coverageValues[i] /= (double) n;
                fitnessValues[i] /= (double) n;
            }

            String outFile = outputDir;
            if (!outFile.endsWith("/")) {
                outFile += "/";
            }
            outFile = outFile + "AVG_ON_ALL_" + type;
            BufferedWriter out = new BufferedWriter(new FileWriter(outFile));

            out.write(firstLine);
            out.newLine();
            for (int i = 0; i < coverageValues.length; i++) {
                out.write(Double.toString(coverageValues[i]) + ";" + Double.toString(fitnessValues[i]));
                out.newLine();
            }
            out.close();
        }
    }

    public int countUncommentedLines(File file) throws Exception {
        BufferedReader in = new BufferedReader(new FileReader(file));
        String line;
        int n = 0;
        while ((line = in.readLine()) != null) {
            line = line.trim();
            if (!line.isEmpty() && !line.startsWith("#")) {
                n++;
            }
        }
        in.close();
        return n;
    }

    public int minUncommentedLinesCount(File[] files) throws Exception {
        int min = countUncommentedLines(files[0]);

        for (int i = 1; i < files.length; i++) {
            int count = countUncommentedLines(files[i]);
            if (count < min) {
                min = count;
            }
        }

        return min;
    }

    public List<Product> loadProductsFromCSVFile(File csvFile, Map<String, Integer> featuresMapRev) throws Exception {
        List<Product> products = new ArrayList<Product>();
        BufferedReader in = new BufferedReader(new FileReader(csvFile));
        String line;
        boolean firstLine = true;
        List<String> features = null;

        if (featuresMapRev != null) {
            features = new ArrayList<String>();
        }
        while ((line = in.readLine()) != null) {
            StringTokenizer tokenizer = new StringTokenizer(line, ";");
            if (firstLine) {
                if (featuresMapRev != null) {
                    while (tokenizer.hasMoreTokens()) {
                        features.add(tokenizer.nextToken().trim());
                    }
                }
                firstLine = false;
            } else {
                Product product = new Product();
                int count;
                if (featuresMapRev != null) {
                    count = 0;
                } else {
                    count = 1;
                }
                while (tokenizer.hasMoreTokens()) {
                    if (tokenizer.nextToken().equals("X")) {
                        if (featuresMapRev != null) {
                            product.add(featuresMapRev.get(features.get(count)));
                        } else {
                            product.add(count);
                        }
                    } else {
                        if (featuresMapRev != null) {
                            product.add(-featuresMapRev.get(features.get(count)));
                        } else {
                            product.add(-count);
                        }
                    }
                    count++;

                }
                products.add(product);
            }
        }
        return products;
    }

    public double[] computeFitnessSums(List<Product> products, int distanceMethod) {
        int size = products.size();

        double[][] distancesMatrix = new double[size][size];

        for (int i = 0; i < size; i++) {
            for (int j = 0; j < size; j++) {
                if (j > i) {
                    distancesMatrix[i][j] = DistancesUtil.getJaccardDistance(products.get(i), products.get(j));
                }
            }
        }
        double[] fitnessSums = new double[size];
        int n = size - 1;

        while (n >= 0) {
            fitnessSums[n] = SimilarityTechnique.getJaccardFitnessSum(distancesMatrix, n + 1);
            n--;
        }
        return fitnessSums;


    }

    public void computeValidPairsToFile(String fmFile, boolean dimacs, int nbParts, int part) throws Exception {

        if (!new File(fmFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }


        int timeoutS = 300;
        List<Integer> featuresList = new ArrayList<Integer>();
        Map<Integer, String> featuresMap = new HashMap<Integer, String>();
        Map<String, Integer> featuresMapRev = new HashMap<String, Integer>();
        if (!dimacs) {

            fm = loadFeatureModel(fmFile);
            fm.loadModel();
            reasonerSAT = new FMReasoningWithSAT("MiniSAT", fm, timeoutS);
            reasonerSAT.init();
            reasonerSAT.getSolver().setTimeout(timeoutS);


            computeFeatures(reasonerSAT, featuresMap, featuresMapRev, featuresList, false, null);

            computeValidPairs(featuresMap, featuresList, (fmFile + ".validpairs"), false, null, nbParts, part);
        } else {
            computeFeatures(null, featuresMap, featuresMapRev, featuresList, true, fmFile);
            dimacsSolver = SolverFactory.instance().createSolverByName("MiniSAT");
            dimacsSolver.setTimeout(timeoutS);
            DimacsReader dr = new DimacsReader(dimacsSolver);
            dr.parseInstance(new FileReader(fmFile));
            computeValidPairs(featuresMap, featuresList, (fmFile + ".validpairs"), true, dimacsSolver, nbParts, part);
        }
    }

    public Set<FeaturesPair> loadPairs(String pairsFile) throws Exception {
        if (!new File(pairsFile).exists()) {
            throw new ParameterException("The specified FM file does not exist");
        }

        LineNumberReader lnr = new LineNumberReader(new FileReader(pairsFile));
        lnr.skip(Long.MAX_VALUE);


        List<FeaturesPair> pairs = new ArrayList<FeaturesPair>(lnr.getLineNumber());

        BufferedReader in = new BufferedReader(new FileReader(pairsFile));
        String line;

        while ((line = in.readLine()) != null) {
            if (!line.isEmpty()) {
                StringTokenizer st = new StringTokenizer(line, ";");

                FeaturesPair pair = new FeaturesPair(Integer.parseInt(st.nextToken()), Integer.parseInt(st.nextToken()));
                pairs.add(pair);
            }
        }

        in.close();

        Set<FeaturesPair> pairsSet = new HashSet<FeaturesPair>(pairs);
        return pairsSet;

    }

    private class CommandLineParser {

        private JCommander jCommander;
        private PrioritizationSolverProducts commandPrioritizationSolverProducts;
        private AverageDataFiles commandAverageDataFiles;
        private PrioritizationSPLCATProducts commandPrioritizationSPLCATProducts;
        private PrioritizationSPLCATSolverProducts commandPrioritizationSPLCATSolverProducts;
        private NormalizeDataFiles commandNormalizeDataFiles;
        private GenerateGA commandGenerateGA;
        private ComputePairs commandComputePairs;
        private ComputeStats commandComputeStats;
        private String[] args;
        public static final String PRIORITIZATION_SOLVER_PRODUCTS = "prio_solver_products";
        public static final String AVERAGE_NORMALIZED_DATA_FILES = "average_data_files";
        public static final String PRIORITIZATION_SPLCAT_PRODUCTS = "prio_splcat_products";
        public static final String PRIORITIZATION_SPLCAT_SOLVER_PRODUCTS = "prio_splcat_solver_products";
        public static final String NORMALIZE_DATA_FILES = "normalize_data_files";
        public static final String GENERATE_GA = "generate_prods_ga";
        public static final String COMPUTE_PAIRS = "compute_valid_pairs";
        public static final String COMPUTE_STATS = "compute_stats";

        public CommandLineParser(String[] args, String programName) {
            this.args = args;
            commandPrioritizationSolverProducts = new PrioritizationSolverProducts();
            commandAverageDataFiles = new AverageDataFiles();
            commandPrioritizationSPLCATProducts = new PrioritizationSPLCATProducts();
            commandPrioritizationSPLCATSolverProducts = new PrioritizationSPLCATSolverProducts();
            commandNormalizeDataFiles = new NormalizeDataFiles();
            commandGenerateGA = new GenerateGA();
            commandComputePairs = new ComputePairs();
            commandComputeStats = new ComputeStats();
            jCommander = new JCommander();
            jCommander.addCommand(PRIORITIZATION_SPLCAT_PRODUCTS, commandPrioritizationSPLCATProducts);
            jCommander.addCommand(PRIORITIZATION_SOLVER_PRODUCTS, commandPrioritizationSolverProducts);
            jCommander.addCommand(PRIORITIZATION_SPLCAT_SOLVER_PRODUCTS, commandPrioritizationSPLCATSolverProducts);
            jCommander.addCommand(AVERAGE_NORMALIZED_DATA_FILES, commandAverageDataFiles);
            jCommander.addCommand(NORMALIZE_DATA_FILES, commandNormalizeDataFiles);
            jCommander.addCommand(GENERATE_GA, commandGenerateGA);
            jCommander.addCommand(COMPUTE_PAIRS, commandComputePairs);
            jCommander.addCommand(COMPUTE_STATS, commandComputeStats);
            jCommander.setProgramName("java -jar " + programName + ".jar");
        }

        public void parseArgs() {
            jCommander.parse(args);
        }

        public void printUsage() {
            jCommander.usage();
        }

        public String getCommandName() {
            return jCommander.getParsedCommand();
        }

        public PrioritizationSolverProducts getCommandPrioritizationSolverProducts() {
            return commandPrioritizationSolverProducts;
        }

        public AverageDataFiles getCommandAverageDataFiles() {
            return commandAverageDataFiles;
        }

        public PrioritizationSPLCATProducts getCommandPrioritizationSPLCATProducts() {
            return commandPrioritizationSPLCATProducts;
        }

        public PrioritizationSPLCATSolverProducts getCommandPrioritizationSPLCATAndSolverProducts() {
            return commandPrioritizationSPLCATSolverProducts;
        }

        public NormalizeDataFiles getCommandNormalizeDataFiles() {
            return commandNormalizeDataFiles;
        }

        public GenerateGA getCommandGenerateGA() {
            return commandGenerateGA;
        }

        public ComputePairs getCommandComputePairs() {
            return commandComputePairs;
        }

        public ComputeStats getCommandComputeStats() {
            return commandComputeStats;
        }

        @Parameters(commandDescription = "Computes the values of the prioritization techniques and fitness function using the "
        + "set of valid products from SPLCAT. ")
        private class PrioritizationSPLCATProducts {

            @Parameter(names = "-csv", description = "CSV file output from SPLCAT tool", required = true)
            private String csvFile;
            @Parameter(names = "-fm", description = "Feature model (SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-o", description = "Output directory", required = true)
            private String outputFile;
            @Parameter(names = "-runs", description = "Number of runs to average the random technique")
            private int runs = 10;
            @Parameter(names = "-p", description = "Valid pairs file")
            private String validPairsFile;
        }

        @Parameters(commandDescription = "Computes the values of the prioritization techniques and fitness function using the "
        + "set of valid products from SPLCAT + additional products from the solver. ")
        private class PrioritizationSPLCATSolverProducts {

            @Parameter(names = "-csv", description = "CSV file output from SPLCAT tool", required = true)
            private String csvFile;
            @Parameter(names = "-fm", description = "Feature model (SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-o", description = "Output directory", required = true)
            private String outputFile;
            @Parameter(names = "-runs", description = "Number of runs to average")
            private int runs = 10;
            @Parameter(names = "-p", description = "Valid pairs file")
            private String validPairsFile;
            @Parameter(names = "-predictable", description = "Use the predictable products for the solver")
            private boolean predictable = false;
        }

        @Parameters(commandDescription = "Computes the values of the prioritization techniques and fitness function using a "
        + "set of products from the solver. The number of products generated will be equal to #features/2")
        private class PrioritizationSolverProducts {

            @Parameter(names = "-fm", description = "Feature model (SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-o", description = "Output directory", required = true)
            private String outputFile;
            @Parameter(names = "-runs", description = "Number of runs to average")
            private int runs = 10;
            @Parameter(names = "-p", description = "Valid pairs file")
            private String validPairsFile;
            @Parameter(names = "-predictable", description = "Use the predictable products for the solver")
            private boolean predictable = false;
        }

        @Parameters(commandDescription = "Compute the average between the data files "
        + "specified in the directory")
        private class AverageDataFiles {

            @Parameter(names = "-i", description = "Directory containing .dat files", required = true)
            private String inputDirectory;
            @Parameter(names = "-o", description = "Output directory")
            private String outputDirectory = "Same as input";
            @Parameter(names = "-noNorm", description = "Don't work on normalized data files. The "
            + "minimum length data file will be used")
            private boolean noNorm = false;
        }

        @Parameters(commandDescription = "Compute the normalized version of the data files "
        + "present specified in the directory")
        private class NormalizeDataFiles {

            @Parameter(names = "-i", description = "Directory containing data file to normalize", required = true)
            private String input;
        }

        @Parameters(commandDescription = "Compute the valid pairs to a file for a XML FM ")
        private class ComputePairs {

            @Parameter(names = "-fm", description = "Feature model (Dimacs or SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-dimacs", description = "Specify if the FM is a dimacs one")
            private boolean dimacs = false;
            @Parameter(names = "-parts", description = "Specify the number of parts")
            private int nbParts = 1;
            @Parameter(names = "-part", description = "Specify which part should be considered. Only if nbParts > 1")
            private int part = 1;
        }

        @Parameters(commandDescription = "Compute some statistics for a FM ")
        private class ComputeStats {

            @Parameter(names = "-fm", description = "Feature model (Dimacs or SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-dimacs", description = "Specify if the FM is a dimacs one")
            private boolean dimacs = false;
        }

        @Parameters(commandDescription = "Generate products using the GA")
        private class GenerateGA {

            @Parameter(names = "-csv", description = "CSV file output from SPLCAT tool. This argument is optional, see -nbProds option")
            private String csvFile;
            @Parameter(names = "-fm", description = "Feature model (SPLOT format)", required = true)
            private String fmFile;
            @Parameter(names = "-o", description = "Output directory", required = true)
            private String outputFile;
            @Parameter(names = "-runs", description = "Number of runs to average")
            private int runs = 10;
            @Parameter(names = "-nbProds", description = "Number of products to generate. If negative, then "
            + "the same number as the SPLCAT output will be used. In this case, the -csv option is required")
            private int nbProds = 10;
            @Parameter(names = "-timeAllowedMS", description = "Time allowed for the GA in ms")
            private long timeAllowed = 60000;
            @Parameter(names = "-p", description = "Valid pairs file")
            private String validPairsFile;
            @Parameter(names = "-dimacs", description = "Specify if the FM is a dimacs one")
            private boolean dimacs = false;
            @Parameter(names = "-noCoverage", description = "Do not compute the coverage (only generate the products)")
            private boolean noCoverage = false;
            @Parameter(names = "-onlyGA", description = "Run only the GA approach")
            private boolean onlyGA = false;
        }
    }

    public static void main(String[] args) {
        SPL.getInstance().parseArgs(args);
    }
}
