/*
 * Author : Christopher Henard (christopher.henard@uni.lu)
 * Date : 21/05/2012
 * Copyright 2012 University of Luxembourg – Interdisciplinary Centre for Security Reliability and Trust (SnT)
 * All rights reserved
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package spl.fm;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * This class represents a product (i.e. a list of features, selected or not).
 * 
 * @author Christopher Henard
 */
public class Product extends HashSet<Integer> {

    /* Relative coverage of this product. This value depends on the number of pairs 
     * covered by the previous products when we evaluate the coverage of a set of 
     * products.     */
    private double coverage;

    public Product() {
        super();
        coverage = 0;
    }

    public Product(Product other) {
        super(other);
    }

    public double getCoverage() {
        return coverage;
    }

    public void setCoverage(double coverage) {
        this.coverage = coverage;
    }

    /**
     * Return the set of pairs of features covered by this product.
     * @return a set of pairs of features covered by this product.
     */
    public Set<FeaturesPair> computeCoveredPairs() {
        int size = size();
        Set<FeaturesPair> pairs = new HashSet<FeaturesPair>(size * (size - 1) / 2);
        Iterator<Integer> it1 = iterator();
        while (it1.hasNext()) {
            int left = it1.next();
            Iterator<Integer> it2 = iterator();
            while (it2.hasNext()) {
                int right = it2.next();
                if (Math.abs(left) != Math.abs(right)) {
                    FeaturesPair p = new FeaturesPair(left, right);
                    pairs.add(p);
                }
            }
        }
        return pairs;
    }
}