#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#if defined HAVE_SETRLIMIT
# include <sys/time.h>
# include <sys/resource.h>
#endif
#include "mbsupport.h"
#include <wchar.h>
#include <wctype.h>
#include <fcntl.h>
#include <stdio.h>
#include "system.h"
#include "argmatch.h"
#include "c-ctype.h"
#include "closeout.h"
#include "error.h"
#include "exclude.h"
#include "exitfail.h"
#include "getopt.h"
#include "grep.h"
#include "intprops.h"
#include "isdir.h"
#include "progname.h"
#include "propername.h"
#include "savedir.h"
#include "version-etc.h"
#include "xalloc.h"
#include "xstrtol.h"
#define SEP_CHAR_SELECTED ':'
#define SEP_CHAR_REJECTED '-'
#define SEP_STR_GROUP    "--"
#define STREQ(a, b) (strcmp (a, b) == 0)
#define AUTHORS \
  proper_name ("Mike Haertel"), \
  _("others, see <http://git.sv.gnu.org/cgit/grep.git/tree/AUTHORS>")
struct stats
{
  struct stats const *parent;
  struct stat stat;
};
static struct stats stats_base;
static int show_help;
static int show_version;
static int suppress_errors;
static int color_option;
static int only_matching;
static int align_tabs;
static const char *group_separator = SEP_STR_GROUP;
static const char *selected_match_color = "01;31";	
static const char *context_match_color  = "01;31";	
static const char *filename_color = "35";	
static const char *line_num_color = "32";	
static const char *byte_num_color = "32";	
static const char *sep_color      = "36";	
static const char *selected_line_color = "";	
static const char *context_line_color  = "";	
static const char *sgr_start = "\33[%sm\33[K";
#define SGR_START  sgr_start
static const char *sgr_end   = "\33[m\33[K";
#define SGR_END    sgr_end
#define PR_SGR_FMT(fmt, s) do { if (*(s)) printf((fmt), (s)); } while (0)
#define PR_SGR_FMT_IF(fmt, s) \
  do { if (color_option && *(s)) printf((fmt), (s)); } while (0)
#define PR_SGR_START(s)    PR_SGR_FMT(   SGR_START, (s))
#define PR_SGR_END(s)      PR_SGR_FMT(   SGR_END,   (s))
#define PR_SGR_START_IF(s) PR_SGR_FMT_IF(SGR_START, (s))
#define PR_SGR_END_IF(s)   PR_SGR_FMT_IF(SGR_END,   (s))
struct color_cap
  {
    const char *name;
    const char **var;
    const char *(*fct)(void);
  };
static const char *
color_cap_mt_fct(void)
{
  context_match_color = selected_match_color;
  return NULL;
}
static const char *
color_cap_rv_fct(void)
{
  color_option = -1;  
  return NULL;
}
static const char *
color_cap_ne_fct(void)
{
  sgr_start = "\33[%sm";
  sgr_end   = "\33[m";
  return NULL;
}
static struct color_cap color_dict[] =
  {
    { "mt", &selected_match_color, color_cap_mt_fct }, 
    { "ms", &selected_match_color, NULL }, 
    { "mc", &context_match_color,  NULL }, 
    { "fn", &filename_color,       NULL }, 
    { "ln", &line_num_color,       NULL }, 
    { "bn", &byte_num_color,       NULL }, 
    { "se", &sep_color,            NULL }, 
    { "sl", &selected_line_color,  NULL }, 
    { "cx", &context_line_color,   NULL }, 
    { "rv", NULL,                  color_cap_rv_fct }, 
    { "ne", NULL,                  color_cap_ne_fct }, 
    { NULL, NULL,                  NULL }
  };
static struct exclude *excluded_patterns;
static struct exclude *included_patterns;
static struct exclude *excluded_directory_patterns;
static char const short_options[] =
"0123456789A:B:C:D:EFGHIPTUVX:abcd:e:f:hiKLlm:noqRrsuvwxyZz";
enum
{
  BINARY_FILES_OPTION = CHAR_MAX + 1,
  COLOR_OPTION,
  INCLUDE_OPTION,
  EXCLUDE_OPTION,
  EXCLUDE_FROM_OPTION,
  LINE_BUFFERED_OPTION,
  LABEL_OPTION,
  EXCLUDE_DIRECTORY_OPTION,
  GROUP_SEPARATOR_OPTION,
  MMAP_OPTION
};
static struct option const long_options[] =
{
  {"basic-regexp",    no_argument, NULL, 'G'},
  {"extended-regexp", no_argument, NULL, 'E'},
  {"fixed-regexp",    no_argument, NULL, 'F'},
  {"fixed-strings",   no_argument, NULL, 'F'},
  {"perl-regexp",     no_argument, NULL, 'P'},
  {"after-context", required_argument, NULL, 'A'},
  {"before-context", required_argument, NULL, 'B'},
  {"binary-files", required_argument, NULL, BINARY_FILES_OPTION},
  {"byte-offset", no_argument, NULL, 'b'},
  {"context", required_argument, NULL, 'C'},
  {"color", optional_argument, NULL, COLOR_OPTION},
  {"colour", optional_argument, NULL, COLOR_OPTION},
  {"count", no_argument, NULL, 'c'},
  {"devices", required_argument, NULL, 'D'},
  {"directories", required_argument, NULL, 'd'},
  {"exclude", required_argument, NULL, EXCLUDE_OPTION},
  {"exclude-from", required_argument, NULL, EXCLUDE_FROM_OPTION},
  {"exclude-dir", required_argument, NULL, EXCLUDE_DIRECTORY_OPTION},
  {"file", required_argument, NULL, 'f'},
  {"files-with-matches", no_argument, NULL, 'l'},
  {"files-without-match", no_argument, NULL, 'L'},
  {"group-separator", required_argument, NULL, GROUP_SEPARATOR_OPTION},
  {"help", no_argument, &show_help, 1},
  {"include", required_argument, NULL, INCLUDE_OPTION},
  {"ignore-case", no_argument, NULL, 'i'},
  {"initial-tab", no_argument, NULL, 'T'},
  {"label", required_argument, NULL, LABEL_OPTION},
  {"line-buffered", no_argument, NULL, LINE_BUFFERED_OPTION},
  {"line-number", no_argument, NULL, 'n'},
  {"line-regexp", no_argument, NULL, 'x'},
  {"max-count", required_argument, NULL, 'm'},
  {"mmap", no_argument, NULL, MMAP_OPTION},
  {"no-filename", no_argument, NULL, 'h'},
  {"no-group-separator", no_argument, NULL, GROUP_SEPARATOR_OPTION},
  {"no-messages", no_argument, NULL, 's'},
  {"null", no_argument, NULL, 'Z'},
  {"null-data", no_argument, NULL, 'z'},
  {"only-matching", no_argument, NULL, 'o'},
  {"quiet", no_argument, NULL, 'q'},
  {"recursive", no_argument, NULL, 'r'},
  {"recursive", no_argument, NULL, 'R'},
  {"regexp", required_argument, NULL, 'e'},
  {"invert-match", no_argument, NULL, 'v'},
  {"silent", no_argument, NULL, 'q'},
  {"text", no_argument, NULL, 'a'},
  {"binary", no_argument, NULL, 'U'},
  {"unix-byte-offsets", no_argument, NULL, 'u'},
  {"version", no_argument, NULL, 'V'},
  {"with-filename", no_argument, NULL, 'H'},
  {"word-regexp", no_argument, NULL, 'w'},
  {0, 0, 0, 0}
};
int match_icase;
int match_words;
int match_lines;
unsigned char eolbyte;
static char const *filename;
static int errseen;
enum directories_type
  {
    READ_DIRECTORIES = 2,
    RECURSE_DIRECTORIES,
    SKIP_DIRECTORIES
  };
static char const *const directories_args[] =
{
  "read", "recurse", "skip", NULL
};
static enum directories_type const directories_types[] =
{
  READ_DIRECTORIES, RECURSE_DIRECTORIES, SKIP_DIRECTORIES
};
ARGMATCH_VERIFY (directories_args, directories_types);
static enum directories_type directories = READ_DIRECTORIES;
static enum
  {
    READ_DEVICES,
    SKIP_DEVICES
  } devices = READ_DEVICES;
static int grepdir (char const *, struct stats const *);
#if defined HAVE_DOS_FILE_CONTENTS
static inline int undossify_input (char *, size_t);
#endif
static compile_fp_t compile;
static execute_fp_t execute;
static void
suppressible_error (char const *mesg, int errnum)
{
  if (! suppress_errors)
    error (0, errnum, "%s", mesg);
  errseen = 1;
}
static void
context_length_arg (char const *str, int *out)
{
  uintmax_t value;
  if (! (xstrtoumax (str, 0, 10, &value, "") == LONGINT_OK
         && 0 <= (*out = value)
         && *out == value))
    {
      error (EXIT_TROUBLE, 0, "%s: %s", str,
             _("invalid context length argument"));
    }
}
static char *buffer;		
static size_t bufalloc;		
#define INITIAL_BUFSIZE 32768	
static int bufdesc;		
static char *bufbeg;		
static char *buflim;		
static size_t pagesize;		
static off_t bufoffset;		
static off_t after_last_match;	
#define ALIGN_TO(val, alignment) \
  ((size_t) (val) % (alignment) == 0 \
   ? (val) \
   : (val) + ((alignment) - (size_t) (val) % (alignment)))
static int
reset (int fd, char const *file, struct stats *stats)
{
  if (! pagesize)
    {
      pagesize = getpagesize ();
      if (pagesize == 0 || 2 * pagesize + 1 <= pagesize)
        abort ();
      bufalloc = ALIGN_TO (INITIAL_BUFSIZE, pagesize) + pagesize + 1;
      buffer = xmalloc (bufalloc);
    }
  bufbeg = buflim = ALIGN_TO (buffer + 1, pagesize);
  bufbeg[-1] = eolbyte;
  bufdesc = fd;
  if (S_ISREG (stats->stat.st_mode))
    {
      if (file)
        bufoffset = 0;
      else
        {
          bufoffset = lseek (fd, 0, SEEK_CUR);
          if (bufoffset < 0)
            {
              error (0, errno, _("lseek failed"));
              return 0;
            }
        }
    }
  return 1;
}
static int
fillbuf (size_t save, struct stats const *stats)
{
  size_t fillsize = 0;
  int cc = 1;
  char *readbuf;
  size_t readsize;
  size_t saved_offset = buflim - save - buffer;
  if (pagesize <= buffer + bufalloc - buflim)
    {
      readbuf = buflim;
      bufbeg = buflim - save;
    }
  else
    {
      size_t minsize = save + pagesize;
      size_t newsize;
      size_t newalloc;
      char *newbuf;
      for (newsize = bufalloc - pagesize - 1; newsize < minsize; newsize *= 2)
        if (newsize * 2 < newsize || newsize * 2 + pagesize + 1 < newsize * 2)
          xalloc_die ();
      if (S_ISREG (stats->stat.st_mode))
        {
          off_t to_be_read = stats->stat.st_size - bufoffset;
          off_t maxsize_off = save + to_be_read;
          if (0 <= to_be_read && to_be_read <= maxsize_off
              && maxsize_off == (size_t) maxsize_off
              && minsize <= (size_t) maxsize_off
              && (size_t) maxsize_off < newsize)
            newsize = maxsize_off;
        }
      newalloc = newsize + pagesize + 1;
      newbuf = bufalloc < newalloc ? xmalloc (bufalloc = newalloc) : buffer;
      readbuf = ALIGN_TO (newbuf + 1 + save, pagesize);
      bufbeg = readbuf - save;
      memmove (bufbeg, buffer + saved_offset, save);
      bufbeg[-1] = eolbyte;
      if (newbuf != buffer)
        {
          free (buffer);
          buffer = newbuf;
        }
    }
  readsize = buffer + bufalloc - readbuf;
  readsize -= readsize % pagesize;
  if (! fillsize)
    {
      ssize_t bytesread;
      while ((bytesread = read (bufdesc, readbuf, readsize)) < 0
             && errno == EINTR)
        continue;
      if (bytesread < 0)
        cc = 0;
      else
        fillsize = bytesread;
    }
  bufoffset += fillsize;
#if defined HAVE_DOS_FILE_CONTENTS
  if (fillsize)
    fillsize = undossify_input (readbuf, fillsize);
#endif
  buflim = readbuf + fillsize;
  return cc;
}
static enum
{
  BINARY_BINARY_FILES,
  TEXT_BINARY_FILES,
  WITHOUT_MATCH_BINARY_FILES
} binary_files;		
static int filename_mask;	
static int out_quiet;		
static int out_invert;		
static int out_file;		
static int out_line;		
static int out_byte;		
static int out_before;		
static int out_after;		
static int count_matches;	
static int list_files;		
static int no_filenames;	
static off_t max_count;		
static int line_buffered;       
static char *label = NULL;      
static uintmax_t totalcc;	
static char const *lastnl;	
static char const *lastout;	
static uintmax_t totalnl;	
static off_t outleft;		
static int pending;		
static int done_on_match;	
static int exit_on_match;	
#if defined HAVE_DOS_FILE_CONTENTS
# include "dosbuf.c"
#endif
static uintmax_t
add_count (uintmax_t a, uintmax_t b)
{
  uintmax_t sum = a + b;
  if (sum < a)
    error (EXIT_TROUBLE, 0, _("input is too large to count"));
  return sum;
}
static void
nlscan (char const *lim)
{
  size_t newlines = 0;
  char const *beg;
  for (beg = lastnl; beg < lim; beg++)
    {
      beg = memchr (beg, eolbyte, lim - beg);
      if (!beg)
        break;
      newlines++;
    }
  totalnl = add_count (totalnl, newlines);
  lastnl = lim;
}
static void
print_filename (void)
{
  PR_SGR_START_IF(filename_color);
  fputs(filename, stdout);
  PR_SGR_END_IF(filename_color);
}
static void
print_sep (char sep)
{
  PR_SGR_START_IF(sep_color);
  fputc(sep, stdout);
  PR_SGR_END_IF(sep_color);
}
static void
print_offset (uintmax_t pos, int min_width, const char *color)
{
  char buf[sizeof pos * CHAR_BIT];
  char *p = buf + sizeof buf;
  do
    {
      *--p = '0' + pos % 10;
      --min_width;
    }
  while ((pos /= 10) != 0);
  if (align_tabs)
    while (--min_width >= 0)
      *--p = ' ';
  PR_SGR_START_IF(color);
  fwrite (p, 1, buf + sizeof buf - p, stdout);
  PR_SGR_END_IF(color);
}
static void
print_line_head (char const *beg, char const *lim, int sep)
{
  int pending_sep = 0;
  if (out_file)
    {
      print_filename();
      if (filename_mask)
        pending_sep = 1;
      else
        fputc(0, stdout);
    }
  if (out_line)
    {
      if (lastnl < lim)
        {
          nlscan (beg);
          totalnl = add_count (totalnl, 1);
          lastnl = lim;
        }
      if (pending_sep)
        print_sep(sep);
      print_offset (totalnl, 4, line_num_color);
      pending_sep = 1;
    }
  if (out_byte)
    {
      uintmax_t pos = add_count (totalcc, beg - bufbeg);
#if defined HAVE_DOS_FILE_CONTENTS
      pos = dossified_pos (pos);
#endif
      if (pending_sep)
        print_sep(sep);
      print_offset (pos, 6, byte_num_color);
      pending_sep = 1;
    }
  if (pending_sep)
    {
      if (align_tabs)
        fputs("\t\b", stdout);
      print_sep(sep);
    }
}
static const char *
print_line_middle (const char *beg, const char *lim,
                   const char *line_color, const char *match_color)
{
  size_t match_size;
  size_t match_offset;
  const char *cur = beg;
  const char *mid = NULL;
  while (cur < lim
         && ((match_offset = execute(beg, lim - beg, &match_size,
                                     beg + (cur - beg))) != (size_t) -1))
    {
      char const *b = beg + match_offset;
      if (b == lim)
        break;
      if (match_size == 0)
        {
          match_size = 1;
          if (!mid)
            mid = cur;
        }
      else
        {
          if (only_matching)
            print_line_head(b, lim, out_invert ? SEP_CHAR_REJECTED
                                               : SEP_CHAR_SELECTED);
          else
            {
              PR_SGR_START(line_color);
              if (mid)
                {
                  cur = mid;
                  mid = NULL;
                }
              fwrite (cur, sizeof (char), b - cur, stdout);
            }
          PR_SGR_START_IF(match_color);
          fwrite (b, sizeof (char), match_size, stdout);
          PR_SGR_END_IF(match_color);
          if (only_matching)
            fputs("\n", stdout);
        }
      cur = b + match_size;
    }
  if (only_matching)
    cur = lim;
  else if (mid)
    cur = mid;
  return cur;
}
static const char *
print_line_tail (const char *beg, const char *lim, const char *line_color)
{
  size_t  eol_size;
  size_t tail_size;
  eol_size   = (lim > beg && lim[-1] == eolbyte);
  eol_size  += (lim - eol_size > beg && lim[-(1 + eol_size)] == '\r');
  tail_size  =  lim - eol_size - beg;
  if (tail_size > 0)
    {
      PR_SGR_START(line_color);
      fwrite(beg, 1, tail_size, stdout);
      beg += tail_size;
      PR_SGR_END(line_color);
    }
  return beg;
}
static void
prline (char const *beg, char const *lim, int sep)
{
  int matching;
  const char *line_color;
  const char *match_color;
  if (!only_matching)
    print_line_head(beg, lim, sep);
  matching = (sep == SEP_CHAR_SELECTED) ^ !!out_invert;
  if (color_option)
    {
      line_color  = (  (sep == SEP_CHAR_SELECTED)
                     ^ (out_invert && (color_option < 0)))
                  ? selected_line_color  : context_line_color;
      match_color = (sep == SEP_CHAR_SELECTED)
                  ? selected_match_color : context_match_color;
    }
  else
    line_color = match_color = NULL; 
  if (   (only_matching && matching)
      || (color_option  && (*line_color || *match_color)))
    {
      if (matching && (only_matching || *match_color))
        beg = print_line_middle(beg, lim, line_color, match_color);
      if (!only_matching && *line_color)
        beg = print_line_tail(beg, lim, line_color);
    }
  if (!only_matching && lim > beg)
    fwrite (beg, 1, lim - beg, stdout);
  if (ferror (stdout))
    error (0, errno, _("writing output"));
  lastout = lim;
  if (line_buffered)
    fflush (stdout);
}
static void
prpending (char const *lim)
{
  if (!lastout)
    lastout = bufbeg;
  while (pending > 0 && lastout < lim)
    {
      char const *nl = memchr (lastout, eolbyte, lim - lastout);
      size_t match_size;
      --pending;
      if (outleft
          || ((execute(lastout, nl + 1 - lastout,
                       &match_size, NULL) == (size_t) -1)
              == !out_invert))
        prline (lastout, nl + 1, SEP_CHAR_REJECTED);
      else
        pending = 0;
    }
}
static void
prtext (char const *beg, char const *lim, int *nlinesp)
{
  static int used;	
  char const *bp, *p;
  char eol = eolbyte;
  int i, n;
  if (!out_quiet && pending > 0)
    prpending (beg);
  p = beg;
  if (!out_quiet)
    {
      bp = lastout ? lastout : bufbeg;
      for (i = 0; i < out_before; ++i)
        if (p > bp)
          do
            --p;
          while (p[-1] != eol);
      if ((out_before || out_after) && used && p != lastout && group_separator)
        {
          PR_SGR_START_IF(sep_color);
          fputs (group_separator, stdout);
          PR_SGR_END_IF(sep_color);
          fputc('\n', stdout);
        }
      while (p < beg)
        {
          char const *nl = memchr (p, eol, beg - p);
          nl++;
          prline (p, nl, SEP_CHAR_REJECTED);
          p = nl;
        }
    }
  if (nlinesp)
    {
      for (n = 0; p < lim && n < outleft; n++)
        {
          char const *nl = memchr (p, eol, lim - p);
          nl++;
          if (!out_quiet)
            prline (p, nl, SEP_CHAR_SELECTED);
          p = nl;
        }
      *nlinesp = n;
      after_last_match = bufoffset - (buflim - p);
    }
  else
    if (!out_quiet)
      prline (beg, lim, SEP_CHAR_SELECTED);
  pending = out_quiet ? 0 : out_after;
  used = 1;
}
static size_t
do_execute (char const *buf, size_t size, size_t *match_size, char const *start_ptr)
{
  size_t result;
  const char *line_next;
  if (MB_CUR_MAX == 1 || !match_icase)
    return execute(buf, size, match_size, start_ptr);
  for (line_next = buf; line_next < buf + size; )
    {
      const char *line_buf = line_next;
      const char *line_end = memchr (line_buf, eolbyte, (buf + size) - line_buf);
      if (line_end == NULL)
        line_next = line_end = buf + size;
      else
        line_next = line_end + 1;
      if (start_ptr && start_ptr >= line_end)
        continue;
      result = execute (line_buf, line_next - line_buf, match_size, start_ptr);
      if (result != (size_t) -1)
        return (line_buf - buf) + result;
    }
  return (size_t) -1;
}
static int
grepbuf (char const *beg, char const *lim)
{
  int nlines, n;
  char const *p;
  size_t match_offset;
  size_t match_size;
  nlines = 0;
  p = beg;
  while ((match_offset = do_execute(p, lim - p, &match_size,
                                    NULL)) != (size_t) -1)
    {
      char const *b = p + match_offset;
      char const *endp = b + match_size;
      if (b == lim)
        break;
      if (!out_invert)
        {
          prtext (b, endp, (int *) 0);
          nlines++;
          outleft--;
          if (!outleft || done_on_match)
            {
              if (exit_on_match)
                exit (EXIT_SUCCESS);
              after_last_match = bufoffset - (buflim - endp);
              return nlines;
            }
        }
      else if (p < b)
        {
          prtext (p, b, &n);
          nlines += n;
          outleft -= n;
          if (!outleft)
            return nlines;
        }
      p = endp;
    }
  if (out_invert && p < lim)
    {
      prtext (p, lim, &n);
      nlines += n;
      outleft -= n;
    }
  return nlines;
}
static int
grep (int fd, char const *file, struct stats *stats)
{
  int nlines, i;
  int not_text;
  size_t residue, save;
  char oldc;
  char *beg;
  char *lim;
  char eol = eolbyte;
  if (!reset (fd, file, stats))
    return 0;
  if (file && directories == RECURSE_DIRECTORIES
      && S_ISDIR (stats->stat.st_mode))
    {
      if (close (fd) != 0)
        error (0, errno, "%s", file);
      return grepdir (file, stats) - 2;
    }
  totalcc = 0;
  lastout = 0;
  totalnl = 0;
  outleft = max_count;
  after_last_match = 0;
  pending = 0;
  nlines = 0;
  residue = 0;
  save = 0;
  if (! fillbuf (save, stats))
    {
      if (! is_EISDIR (errno, file))
        suppressible_error (filename, errno);
      return 0;
    }
  not_text = (((binary_files == BINARY_BINARY_FILES && !out_quiet)
               || binary_files == WITHOUT_MATCH_BINARY_FILES)
              && memchr (bufbeg, eol ? '\0' : '\200', buflim - bufbeg));
  if (not_text && binary_files == WITHOUT_MATCH_BINARY_FILES)
    return 0;
  done_on_match += not_text;
  out_quiet += not_text;
  for (;;)
    {
      lastnl = bufbeg;
      if (lastout)
        lastout = bufbeg;
      beg = bufbeg + save;
      if (beg == buflim)
        break;
      oldc = beg[-1];
      beg[-1] = eol;
      for (lim = buflim; lim[-1] != eol; lim--)
        continue;
      beg[-1] = oldc;
      if (lim == beg)
        lim = beg - residue;
      beg -= residue;
      residue = buflim - lim;
      if (beg < lim)
        {
          if (outleft)
            nlines += grepbuf (beg, lim);
          if (pending)
            prpending (lim);
          if((!outleft && !pending) || (nlines && done_on_match && !out_invert))
            goto finish_grep;
        }
      i = 0;
      beg = lim;
      while (i < out_before && beg > bufbeg && beg != lastout)
        {
          ++i;
          do
            --beg;
          while (beg[-1] != eol);
        }
      if (beg != lastout)
        lastout = 0;
      save = residue + lim - beg;
      if (out_byte)
        totalcc = add_count (totalcc, buflim - bufbeg - save);
      if (out_line)
        nlscan (beg);
      if (! fillbuf (save, stats))
        {
          if (! is_EISDIR (errno, file))
            suppressible_error (filename, errno);
          goto finish_grep;
        }
    }
  if (residue)
    {
      *buflim++ = eol;
      if (outleft)
        nlines += grepbuf (bufbeg + save - residue, buflim);
      if (pending)
        prpending (buflim);
    }
 finish_grep:
  done_on_match -= not_text;
  out_quiet -= not_text;
  if ((not_text & ~out_quiet) && nlines != 0)
    printf (_("Binary file %s matches\n"), filename);
  return nlines;
}
static int
grepfile (char const *file, struct stats *stats)
{
  int desc;
  int count;
  int status;
  if (! file)
    {
      desc = 0;
      filename = label ? label : _("(standard input)");
    }
  else
    {
      if (stat (file, &stats->stat) != 0)
        {
          suppressible_error (file, errno);
          return 1;
        }
      if (directories == SKIP_DIRECTORIES && S_ISDIR (stats->stat.st_mode))
        return 1;
      if (devices == SKIP_DEVICES && (S_ISCHR (stats->stat.st_mode)
                                      || S_ISBLK (stats->stat.st_mode)
                                      || S_ISSOCK (stats->stat.st_mode)
                                      || S_ISFIFO (stats->stat.st_mode)))
        return 1;
      while ((desc = open (file, O_RDONLY)) < 0 && errno == EINTR)
        continue;
      if (desc < 0)
        {
          int e = errno;
          if (is_EISDIR (e, file) && directories == RECURSE_DIRECTORIES)
            {
              if (stat (file, &stats->stat) != 0)
                {
                  error (0, errno, "%s", file);
                  return 1;
                }
              return grepdir (file, stats);
            }
          if (!suppress_errors)
            {
              if (directories == SKIP_DIRECTORIES)
                switch (e)
                  {
#if defined EISDIR
                  case EISDIR:
                    return 1;
#endif
                  case EACCES:
                    if (isdir (file))
                      return 1;
                    break;
                  }
            }
          suppressible_error (file, e);
          return 1;
        }
      filename = file;
    }
#if defined SET_BINARY
  if (!isatty (desc))
    SET_BINARY (desc);
#endif
  count = grep (desc, file, stats);
  if (count < 0)
    status = count + 2;
  else
    {
      if (count_matches)
        {
          if (out_file)
            {
              print_filename();
              if (filename_mask)
                print_sep(SEP_CHAR_SELECTED);
              else
                fputc(0, stdout);
            }
          printf ("%d\n", count);
        }
      status = !count;
      if (list_files == 1 - 2 * status)
        {
          print_filename();
          fputc('\n' & filename_mask, stdout);
        }
      if (! file)
        {
          off_t required_offset = outleft ? bufoffset : after_last_match;
          if (required_offset != bufoffset
              && lseek (desc, required_offset, SEEK_SET) < 0
              && S_ISREG (stats->stat.st_mode))
            error (0, errno, "%s", filename);
        }
      else
        while (close (desc) != 0)
          if (errno != EINTR)
            {
              error (0, errno, "%s", file);
              break;
            }
    }
  return status;
}
static int
grepdir (char const *dir, struct stats const *stats)
{
  struct stats const *ancestor;
  char *name_space;
  int status = 1;
  if ( excluded_directory_patterns &&
       excluded_file_name (excluded_directory_patterns, dir)  ) {
       return 1;
  }
  if (stats->stat.st_ino)
    for (ancestor = stats;  (ancestor = ancestor->parent) != 0;  )
      if (ancestor->stat.st_ino == stats->stat.st_ino
          && ancestor->stat.st_dev == stats->stat.st_dev)
        {
          if (!suppress_errors)
            error (0, 0, _("warning: %s: %s"), dir,
                   _("recursive directory loop"));
          return 1;
        }
  name_space = savedir (dir, stats->stat.st_size, included_patterns,
                        excluded_patterns, excluded_directory_patterns);
  if (! name_space)
    {
      if (errno)
        suppressible_error (dir, errno);
      else
        xalloc_die ();
    }
  else
    {
      size_t dirlen = strlen (dir);
      int needs_slash = ! (dirlen == FILE_SYSTEM_PREFIX_LEN (dir)
                           || ISSLASH (dir[dirlen - 1]));
      char *file = NULL;
      char const *namep = name_space;
      struct stats child;
      child.parent = stats;
      out_file += !no_filenames;
      while (*namep)
        {
          size_t namelen = strlen (namep);
          file = xrealloc (file, dirlen + 1 + namelen + 1);
          strcpy (file, dir);
          file[dirlen] = '/';
          strcpy (file + dirlen + needs_slash, namep);
          namep += namelen + 1;
          status &= grepfile (file, &child);
        }
      out_file -= !no_filenames;
      free (file);
      free (name_space);
    }
  return status;
}
void usage (int status) __attribute__ ((noreturn));
void
usage (int status)
{
  if (status != 0)
    {
      fprintf (stderr, _("Usage: %s [OPTION]... PATTERN [FILE]...\n"),
               program_name);
      fprintf (stderr, _("Try `%s --help' for more information.\n"),
               program_name);
    }
  else
    {
      printf (_("Usage: %s [OPTION]... PATTERN [FILE]...\n"), program_name);
      printf (_("\
Search for PATTERN in each FILE or standard input.\n"));
      printf ("%s", gettext (before_options));
      printf (_("\
Example: %s -i 'hello world' menu.h main.c\n\
\n\
Regexp selection and interpretation:\n"), program_name);
      if (matchers[1].name)
        printf (_("\
  -E, --extended-regexp     PATTERN is an extended regular expression (ERE)\n\
  -F, --fixed-strings       PATTERN is a set of newline-separated fixed strings\n\
  -G, --basic-regexp        PATTERN is a basic regular expression (BRE)\n\
  -P, --perl-regexp         PATTERN is a Perl regular expression\n"));
      printf (_("\
  -e, --regexp=PATTERN      use PATTERN for matching\n\
  -f, --file=FILE           obtain PATTERN from FILE\n\
  -i, --ignore-case         ignore case distinctions\n\
  -w, --word-regexp         force PATTERN to match only whole words\n\
  -x, --line-regexp         force PATTERN to match only whole lines\n\
  -z, --null-data           a data line ends in 0 byte, not newline\n"));
      printf (_("\
\n\
Miscellaneous:\n\
  -s, --no-messages         suppress error messages\n\
  -v, --invert-match        select non-matching lines\n\
  -V, --version             print version information and exit\n\
      --help                display this help and exit\n\
      --mmap                ignored for backwards compatibility\n"));
      printf (_("\
\n\
Output control:\n\
  -m, --max-count=NUM       stop after NUM matches\n\
  -b, --byte-offset         print the byte offset with output lines\n\
  -n, --line-number         print line number with output lines\n\
      --line-buffered       flush output on every line\n\
  -H, --with-filename       print the filename for each match\n\
  -h, --no-filename         suppress the prefixing filename on output\n\
      --label=LABEL         print LABEL as filename for standard input\n\
"));
      printf (_("\
  -o, --only-matching       show only the part of a line matching PATTERN\n\
  -q, --quiet, --silent     suppress all normal output\n\
      --binary-files=TYPE   assume that binary files are TYPE;\n\
                            TYPE is `binary', `text', or `without-match'\n\
  -a, --text                equivalent to --binary-files=text\n\
"));
      printf (_("\
  -I                        equivalent to --binary-files=without-match\n\
  -d, --directories=ACTION  how to handle directories;\n\
                            ACTION is `read', `recurse', or `skip'\n\
  -D, --devices=ACTION      how to handle devices, FIFOs and sockets;\n\
                            ACTION is `read' or `skip'\n\
  -R, -r, --recursive       equivalent to --directories=recurse\n\
"));
      printf (_("\
      --include=FILE_PATTERN  search only files that match FILE_PATTERN\n\
      --exclude=FILE_PATTERN  skip files and directories matching FILE_PATTERN\n\
      --exclude-from=FILE   skip files matching any file pattern from FILE\n\
      --exclude-dir=PATTERN  directories that match PATTERN will be skipped.\n\
"));
      printf (_("\
  -L, --files-without-match  print only names of FILEs containing no match\n\
  -l, --files-with-matches  print only names of FILEs containing matches\n\
  -c, --count               print only a count of matching lines per FILE\n\
  -T, --initial-tab         make tabs line up (if needed)\n\
  -Z, --null                print 0 byte after FILE name\n"));
      printf (_("\
\n\
Context control:\n\
  -B, --before-context=NUM  print NUM lines of leading context\n\
  -A, --after-context=NUM   print NUM lines of trailing context\n\
  -C, --context=NUM         print NUM lines of output context\n\
"));
      printf (_("\
  -NUM                      same as --context=NUM\n\
      --color[=WHEN],\n\
      --colour[=WHEN]       use markers to highlight the matching strings;\n\
                            WHEN is `always', `never', or `auto'\n\
  -U, --binary              do not strip CR characters at EOL (MSDOS)\n\
  -u, --unix-byte-offsets   report offsets as if CRs were not there (MSDOS)\n\
\n"));
      printf ("%s", _(after_options));
      printf (_("\
With no FILE, or when FILE is -, read standard input.  If less than two FILEs\n\
are given, assume -h.  Exit status is 0 if any line was selected, 1 otherwise;\n\
if any error occurs and -q was not given, the exit status is 2.\n"));
      printf (_("\nReport bugs to: %s\n"), PACKAGE_BUGREPORT);
      printf (_("GNU Grep home page: <%s>\n"),
              "http://www.gnu.org/software/grep/");
      fputs (_("General help using GNU software: <http://www.gnu.org/gethelp/>\n"),
             stdout);
    }
  exit (status);
}
static void
setmatcher (char const *m)
{
  static char const *matcher;
  unsigned int i;
  if (!m)
    {
      compile = matchers[0].compile;
      execute = matchers[0].execute;
      if (!matchers[1].name)
        matcher = matchers[0].name;
    }
  else if (matcher)
    {
      if (matcher && STREQ (matcher, m))
        ;
      else if (!matchers[1].name)
        error (EXIT_TROUBLE, 0, _("%s can only use the %s pattern syntax"),
               program_name, matcher);
      else
        error (EXIT_TROUBLE, 0, _("conflicting matchers specified"));
    }
  else
    {
      for (i = 0; matchers[i].name; i++)
        if (STREQ (m, matchers[i].name))
          {
            compile = matchers[i].compile;
            execute = matchers[i].execute;
            matcher = m;
            return;
          }
      error (EXIT_TROUBLE, 0, _("invalid matcher %s"), m);
    }
}
static void
set_limits(void)
{
#if defined HAVE_SETRLIMIT && defined RLIMIT_STACK
  struct rlimit rlim;
  if (!getrlimit (RLIMIT_STACK, &rlim))
    {
      long newlim;
      extern long int re_max_failures; 
      newlim = re_max_failures * 2 * 20 * sizeof (char *);
      if (newlim > rlim.rlim_max)
        {
          newlim = rlim.rlim_max;
          re_max_failures = newlim / (2 * 20 * sizeof (char *));
        }
      if (rlim.rlim_cur < newlim)
        {
          rlim.rlim_cur = newlim;
          setrlimit (RLIMIT_STACK, &rlim);
        }
    }
#endif
}
static int
prepend_args (char const *options, char *buf, char **argv)
{
  char const *o = options;
  char *b = buf;
  int n = 0;
  for (;;)
    {
      while (c_isspace ((unsigned char) *o))
        o++;
      if (!*o)
        return n;
      if (argv)
        argv[n] = b;
      n++;
      do
        if ((*b++ = *o++) == '\\' && *o)
          b[-1] = *o++;
      while (*o && ! c_isspace ((unsigned char) *o));
      *b++ = '\0';
    }
}
static void
prepend_default_options (char const *options, int *pargc, char ***pargv)
{
  if (options && *options)
    {
      char *buf = xmalloc (strlen (options) + 1);
      int prepended = prepend_args (options, buf, (char **) NULL);
      int argc = *pargc;
      char * const *argv = *pargv;
      char **pp = xmalloc ((prepended + argc + 1) * sizeof *pp);
      *pargc = prepended + argc;
      *pargv = pp;
      *pp++ = *argv++;
      pp += prepend_args (options, buf, pp);
      while ((*pp++ = *argv++))
        continue;
    }
}
static int
get_nondigit_option (int argc, char *const *argv, int *default_context)
{
  static int prev_digit_optind = -1;
  int opt, this_digit_optind, was_digit;
  char buf[sizeof (uintmax_t) * CHAR_BIT + 4];
  char *p = buf;
  was_digit = 0;
  this_digit_optind = optind;
  while (opt = getopt_long (argc, (char **) argv, short_options, long_options,
                            NULL),
         '0' <= opt && opt <= '9')
    {
      if (prev_digit_optind != this_digit_optind || !was_digit)
        {
          p = buf;
        }
      else
        {
          p -= buf[0] == '0';
        }
      if (p == buf + sizeof buf - 4)
        {
          strcpy (p, "...");
          p += 3;
          break;
        }
      *p++ = opt;
      was_digit = 1;
      prev_digit_optind = this_digit_optind;
      this_digit_optind = optind;
    }
  if (p != buf)
    {
      *p = '\0';
      context_length_arg (buf, default_context);
    }
  return opt;
}
static void
parse_grep_colors (void)
{
  const char *p;
  char *q;
  char *name;
  char *val;
  p = getenv("GREP_COLORS"); 
  if (p == NULL || *p == '\0')
    return;
  q = xmalloc(strlen(p) + 1);
  if (q == NULL)
    return;
  strcpy(q, p);
  name = q;
  val = NULL;
  for (;;)
    if (*q == ':' || *q == '\0')
      {
        char c = *q;
        struct color_cap *cap;
        *q++ = '\0'; 
        for (cap = color_dict; cap->name; cap++)
          if (STREQ (cap->name, name))
            break;
        if (cap->name)
          {
            if (cap->var)
              {
                if (val)
              *(cap->var) = val;
                else
              error(0, 0, _("in GREP_COLORS=\"%s\", the \"%s\" capacity "
                                "needs a value (\"=...\"); skipped"), p, name);
          }
        else if (val)
          error(0, 0, _("in GREP_COLORS=\"%s\", the \"%s\" capacity "
                "is boolean and cannot take a value (\"=%s\"); skipped"),
                p, name, val);
      }
        if (cap->fct)
          {
            const char *err_str = cap->fct();
            if (err_str)
              error(0, 0, _("in GREP_COLORS=\"%s\", the \"%s\" capacity %s"),
                    p, name, err_str);
          }
        if (c == '\0')
          return;
        name = q;
        val = NULL;
      }
    else if (*q == '=')
      {
        if (q == name || val)
          goto ill_formed;
        *q++ = '\0'; 
        val = q; 
      }
    else if (val == NULL)
      q++; 
    else if (*q == ';' || (*q >= '0' && *q <= '9'))
      q++; 
    else
      goto ill_formed;
 ill_formed:
  error(0, 0, _("stopped processing of ill-formed GREP_COLORS=\"%s\" "
                "at remaining substring \"%s\""), p, q);
}
int
main (int argc, char **argv)
{
  char *keys;
  size_t keycc, oldcc, keyalloc;
  int with_filenames;
  int opt, cc, status;
  int default_context;
  FILE *fp;
  initialize_main (&argc, &argv);
  set_program_name (argv[0]);
  program_name = argv[0];
  keys = NULL;
  keycc = 0;
  with_filenames = 0;
  eolbyte = '\n';
  filename_mask = ~0;
  max_count = TYPE_MAXIMUM (off_t);
  out_after = out_before = -1;
  default_context = 0;
  only_matching = 0;
#if defined HAVE_SETLOCALE
  setlocale (LC_ALL, "");
#endif
#if defined ENABLE_NLS
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#endif
  exit_failure = EXIT_TROUBLE;
  atexit (close_stdout);
  prepend_default_options (getenv ("GREP_OPTIONS"), &argc, &argv);
  setmatcher (NULL);
  while ((opt = get_nondigit_option (argc, argv, &default_context)) != -1)
    switch (opt)
      {
      case 'A':
        context_length_arg (optarg, &out_after);
        break;
      case 'B':
        context_length_arg (optarg, &out_before);
        break;
      case 'C':
        context_length_arg (optarg, &default_context);
        break;
      case 'D':
        if (STREQ (optarg, "read"))
          devices = READ_DEVICES;
        else if (STREQ (optarg, "skip"))
          devices = SKIP_DEVICES;
        else
          error (EXIT_TROUBLE, 0, _("unknown devices method"));
        break;
      case 'E':
        setmatcher ("egrep");
        break;
      case 'F':
        setmatcher ("fgrep");
        break;
      case 'P':
        setmatcher ("perl");
        break;
      case 'G':
        setmatcher ("grep");
        break;
      case 'X': 
        setmatcher (optarg);
        break;
      case 'H':
        with_filenames = 1;
        no_filenames = 0;
        break;
      case 'I':
        binary_files = WITHOUT_MATCH_BINARY_FILES;
        break;
      case 'T':
        align_tabs = 1;
        break;
      case 'U':
#if defined HAVE_DOS_FILE_CONTENTS
        dos_use_file_type = DOS_BINARY;
#endif
        break;
      case 'u':
#if defined HAVE_DOS_FILE_CONTENTS
        dos_report_unix_offset = 1;
#endif
        break;
      case 'V':
        show_version = 1;
        break;
      case 'a':
        binary_files = TEXT_BINARY_FILES;
        break;
      case 'b':
        out_byte = 1;
        break;
      case 'c':
        count_matches = 1;
        break;
      case 'd':
        directories = XARGMATCH ("--directories", optarg,
                                 directories_args, directories_types);
        break;
      case 'e':
        cc = strlen (optarg);
        keys = xrealloc (keys, keycc + cc + 1);
        strcpy (&keys[keycc], optarg);
        keycc += cc;
        keys[keycc++] = '\n';
        break;
      case 'f':
        fp = STREQ (optarg, "-") ? stdin : fopen (optarg, "r");
        if (!fp)
          error (EXIT_TROUBLE, errno, "%s", optarg);
        for (keyalloc = 1; keyalloc <= keycc + 1; keyalloc *= 2)
          ;
        keys = xrealloc (keys, keyalloc);
        oldcc = keycc;
        while (!feof (fp)
               && (cc = fread (keys + keycc, 1, keyalloc - 1 - keycc, fp)) > 0)
          {
            keycc += cc;
            if (keycc == keyalloc - 1)
              keys = xrealloc (keys, keyalloc *= 2);
          }
        if (fp != stdin)
          fclose(fp);
        if (oldcc != keycc && keys[keycc - 1] != '\n')
          keys[keycc++] = '\n';
        break;
      case 'h':
        with_filenames = 0;
        no_filenames = 1;
        break;
      case 'i':
      case 'y':			
        match_icase = 1;
        break;
      case 'L':
        list_files = -1;
        break;
      case 'l':
        list_files = 1;
        break;
      case 'm':
        {
          uintmax_t value;
          switch (xstrtoumax (optarg, 0, 10, &value, ""))
            {
            case LONGINT_OK:
              max_count = value;
              if (0 <= max_count && max_count == value)
                break;
            case LONGINT_OVERFLOW:
              max_count = TYPE_MAXIMUM (off_t);
              break;
            default:
              error (EXIT_TROUBLE, 0, _("invalid max count"));
            }
        }
        break;
      case 'n':
        out_line = 1;
        break;
      case 'o':
        only_matching = 1;
        break;
      case 'q':
        exit_on_match = 1;
        exit_failure = 0;
        break;
      case 'R':
      case 'r':
        directories = RECURSE_DIRECTORIES;
        break;
      case 's':
        suppress_errors = 1;
        break;
      case 'v':
        out_invert = 1;
        break;
      case 'w':
        match_words = 1;
        break;
      case 'x':
        match_lines = 1;
        break;
      case 'Z':
        filename_mask = 0;
        break;
      case 'z':
        eolbyte = '\0';
        break;
      case BINARY_FILES_OPTION:
        if (STREQ (optarg, "binary"))
          binary_files = BINARY_BINARY_FILES;
        else if (STREQ (optarg, "text"))
          binary_files = TEXT_BINARY_FILES;
        else if (STREQ (optarg, "without-match"))
          binary_files = WITHOUT_MATCH_BINARY_FILES;
        else
          error (EXIT_TROUBLE, 0, _("unknown binary-files type"));
        break;
      case COLOR_OPTION:
        if(optarg) {
          if(!strcasecmp(optarg, "always") || !strcasecmp(optarg, "yes") ||
             !strcasecmp(optarg, "force"))
            color_option = 1;
          else if(!strcasecmp(optarg, "never") || !strcasecmp(optarg, "no") ||
                  !strcasecmp(optarg, "none"))
            color_option = 0;
          else if(!strcasecmp(optarg, "auto") || !strcasecmp(optarg, "tty") ||
                  !strcasecmp(optarg, "if-tty"))
            color_option = 2;
          else
            show_help = 1;
        } else
          color_option = 2;
        if (color_option == 2)
          {
            char const *t;
            if (isatty (STDOUT_FILENO) && (t = getenv ("TERM"))
                && !STREQ (t, "dumb"))
              color_option = 1;
            else
              color_option = 0;
          }
        break;
      case EXCLUDE_OPTION:
        if (!excluded_patterns)
          excluded_patterns = new_exclude ();
        add_exclude (excluded_patterns, optarg, EXCLUDE_WILDCARDS);
        break;
      case EXCLUDE_FROM_OPTION:
        if (!excluded_patterns)
          excluded_patterns = new_exclude ();
        if (add_exclude_file (add_exclude, excluded_patterns, optarg,
                              EXCLUDE_WILDCARDS, '\n') != 0)
          {
            error (EXIT_TROUBLE, errno, "%s", optarg);
          }
        break;
      case EXCLUDE_DIRECTORY_OPTION:
        if (!excluded_directory_patterns)
          excluded_directory_patterns = new_exclude ();
        add_exclude (excluded_directory_patterns, optarg, EXCLUDE_WILDCARDS);
        break;
      case INCLUDE_OPTION:
        if (!included_patterns)
          included_patterns = new_exclude ();
        add_exclude (included_patterns, optarg,
                     EXCLUDE_WILDCARDS | EXCLUDE_INCLUDE);
        break;
      case GROUP_SEPARATOR_OPTION:
        group_separator = optarg;
        break;
      case LINE_BUFFERED_OPTION:
        line_buffered = 1;
        break;
      case LABEL_OPTION:
        label = optarg;
        break;
      case MMAP_OPTION:
      case 0:
        break;
      default:
        usage (EXIT_TROUBLE);
        break;
      }
  if (exit_on_match)
    list_files = 0;
  if (exit_on_match | list_files)
    {
      count_matches = 0;
      done_on_match = 1;
    }
  out_quiet = count_matches | done_on_match;
  if (out_after < 0)
    out_after = default_context;
  if (out_before < 0)
    out_before = default_context;
  if (color_option)
    {
      char *userval = getenv ("GREP_COLOR");
      if (userval != NULL && *userval != '\0')
        selected_match_color = context_match_color = userval;
      parse_grep_colors();
    }
  if (show_version)
    {
      version_etc (stdout, program_name, PACKAGE_NAME, VERSION, AUTHORS, \
                   (char *) NULL);					\
      exit (EXIT_SUCCESS);						\
    }
  if (show_help)
    usage (EXIT_SUCCESS);
  if (keys)
    {
      if (keycc == 0)
        {
          out_invert ^= 1;
          match_lines = match_words = 0;
        }
      else
        --keycc;
    }
  else
    if (optind < argc)
      {
        keycc = strlen(argv[optind]);
        keys = xmalloc(keycc + 1);
        strcpy(keys, argv[optind++]);
      }
    else
      usage (EXIT_TROUBLE);
  set_limits();
  compile(keys, keycc);
  free (keys);
  if ((argc - optind > 1 && !no_filenames) || with_filenames)
    out_file = 1;
#ifdef SET_BINARY
  if (!isatty (1))
    SET_BINARY (1);
#endif
  if (max_count == 0)
    exit (EXIT_FAILURE);
  if (optind < argc)
    {
        status = 1;
        do
        {
          char *file = argv[optind];
          if ((included_patterns || excluded_patterns)
              && !isdir (file))
            {
              if (included_patterns
                  && excluded_file_name (included_patterns, file))
                continue;
              if (excluded_patterns
                  && excluded_file_name (excluded_patterns, file))
                continue;
            }
          status &= grepfile (STREQ (file, "-") ? (char *) NULL : file,
                              &stats_base);
        }
        while ( ++optind < argc);
    }
  else
    status = grepfile ((char *) NULL, &stats_base);
  exit (errseen ? EXIT_TROUBLE : status);
}
