#include <config.h>
#include <string.h>
#include <stdbool.h>
#include <stddef.h>  
#include "malloca.h"
#include "mbuiter.h"
#define CANON_ELEMENT(c) c
#include "str-kmp.h"
static bool
knuth_morris_pratt_multibyte (const char *haystack, const char *needle,
                              const char **resultp)
{
  size_t m = mbslen (needle);
  mbchar_t *needle_mbchars;
  size_t *table;
  char *memory = (char *) nmalloca (m, sizeof (mbchar_t) + sizeof (size_t));
  if (memory == NULL)
    return false;
  needle_mbchars = (mbchar_t *) memory;
  table = (size_t *) (memory + m * sizeof (mbchar_t));
  {
    mbui_iterator_t iter;
    size_t j;
    j = 0;
    for (mbui_init (iter, needle); mbui_avail (iter); mbui_advance (iter), j++)
      mb_copy (&needle_mbchars[j], &mbui_cur (iter));
  }
  {
    size_t i, j;
    table[1] = 1;
    j = 0;
    for (i = 2; i < m; i++)
      {
        mbchar_t *b = &needle_mbchars[i - 1];
        for (;;)
          {
            if (mb_equal (*b, needle_mbchars[j]))
              {
                table[i] = i - ++j;
                break;
              }
            if (j == 0)
              {
                table[i] = i;
                break;
              }
            j = j - table[j];
          }
      }
  }
  {
    size_t j;
    mbui_iterator_t rhaystack;
    mbui_iterator_t phaystack;
    *resultp = NULL;
    j = 0;
    mbui_init (rhaystack, haystack);
    mbui_init (phaystack, haystack);
    while (mbui_avail (phaystack))
      if (mb_equal (needle_mbchars[j], mbui_cur (phaystack)))
        {
          j++;
          mbui_advance (phaystack);
          if (j == m)
            {
              *resultp = mbui_cur_ptr (rhaystack);
              break;
            }
        }
      else if (j > 0)
        {
          size_t count = table[j];
          j -= count;
          for (; count > 0; count--)
            {
              if (!mbui_avail (rhaystack))
                abort ();
              mbui_advance (rhaystack);
            }
        }
      else
        {
          if (!mbui_avail (rhaystack))
            abort ();
          mbui_advance (rhaystack);
          mbui_advance (phaystack);
        }
  }
  freea (memory);
  return true;
}
char *
mbsstr (const char *haystack, const char *needle)
{
  if (MB_CUR_MAX > 1)
    {
      mbui_iterator_t iter_needle;
      mbui_init (iter_needle, needle);
      if (mbui_avail (iter_needle))
        {
          bool try_kmp = true;
          size_t outer_loop_count = 0;
          size_t comparison_count = 0;
          size_t last_ccount = 0;                  
          mbui_iterator_t iter_needle_last_ccount; 
          mbui_iterator_t iter_haystack;
          mbui_init (iter_needle_last_ccount, needle);
          mbui_init (iter_haystack, haystack);
          for (;; mbui_advance (iter_haystack))
            {
              if (!mbui_avail (iter_haystack))
                return NULL;
              if (try_kmp
                  && outer_loop_count >= 10
                  && comparison_count >= 5 * outer_loop_count)
                {
                  size_t count = comparison_count - last_ccount;
                  for (;
                       count > 0 && mbui_avail (iter_needle_last_ccount);
                       count--)
                    mbui_advance (iter_needle_last_ccount);
                  last_ccount = comparison_count;
                  if (!mbui_avail (iter_needle_last_ccount))
                    {
                      const char *result;
                      bool success =
                        knuth_morris_pratt_multibyte (haystack, needle,
                                                      &result);
                      if (success)
                        return (char *) result;
                      try_kmp = false;
                    }
                }
              outer_loop_count++;
              comparison_count++;
              if (mb_equal (mbui_cur (iter_haystack), mbui_cur (iter_needle)))
                {
                  mbui_iterator_t rhaystack;
                  mbui_iterator_t rneedle;
                  memcpy (&rhaystack, &iter_haystack, sizeof (mbui_iterator_t));
                  mbui_advance (rhaystack);
                  mbui_init (rneedle, needle);
                  if (!mbui_avail (rneedle))
                    abort ();
                  mbui_advance (rneedle);
                  for (;; mbui_advance (rhaystack), mbui_advance (rneedle))
                    {
                      if (!mbui_avail (rneedle))
                        return (char *) mbui_cur_ptr (iter_haystack);
                      if (!mbui_avail (rhaystack))
                        return NULL;
                      comparison_count++;
                      if (!mb_equal (mbui_cur (rhaystack), mbui_cur (rneedle)))
                        break;
                    }
                }
            }
        }
      else
        return (char *) haystack;
    }
  else
    {
      if (*needle != '\0')
        {
          bool try_kmp = true;
          size_t outer_loop_count = 0;
          size_t comparison_count = 0;
          size_t last_ccount = 0;                  
          const char *needle_last_ccount = needle; 
          char b = *needle++;
          for (;; haystack++)
            {
              if (*haystack == '\0')
                return NULL;
              if (try_kmp
                  && outer_loop_count >= 10
                  && comparison_count >= 5 * outer_loop_count)
                {
                  if (needle_last_ccount != NULL)
                    {
                      needle_last_ccount +=
                        strnlen (needle_last_ccount,
                                 comparison_count - last_ccount);
                      if (*needle_last_ccount == '\0')
                        needle_last_ccount = NULL;
                      last_ccount = comparison_count;
                    }
                  if (needle_last_ccount == NULL)
                    {
                      const char *result;
                      bool success =
                        knuth_morris_pratt_unibyte (haystack, needle - 1,
                                                    &result);
                      if (success)
                        return (char *) result;
                      try_kmp = false;
                    }
                }
              outer_loop_count++;
              comparison_count++;
              if (*haystack == b)
                {
                  const char *rhaystack = haystack + 1;
                  const char *rneedle = needle;
                  for (;; rhaystack++, rneedle++)
                    {
                      if (*rneedle == '\0')
                        return (char *) haystack;
                      if (*rhaystack == '\0')
                        return NULL;
                      comparison_count++;
                      if (*rhaystack != *rneedle)
                        break;
                    }
                }
            }
        }
      else
        return (char *) haystack;
    }
}
