#include <config.h>
#include "striconv.h"
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_ICONV
# include <iconv.h>
# include <limits.h>
#endif
#include "c-strcase.h"
#ifndef SIZE_MAX
# define SIZE_MAX ((size_t) -1)
#endif
#if HAVE_ICONV
int
mem_cd_iconv (const char *src, size_t srclen, iconv_t cd,
              char **resultp, size_t *lengthp)
{
# define tmpbufsize 4096
  size_t length;
  char *result;
# if defined _LIBICONV_VERSION \
     || !((__GLIBC__ - 0 == 2 && __GLIBC_MINOR__ - 0 <= 1) || defined __sun)
  iconv (cd, NULL, NULL, NULL, NULL);
# endif
  {
    size_t count = 0;
    union { unsigned int align; char buf[tmpbufsize]; } tmp;
# define tmpbuf tmp.buf
    const char *inptr = src;
    size_t insize = srclen;
    while (insize > 0)
      {
        char *outptr = tmpbuf;
        size_t outsize = tmpbufsize;
        size_t res = iconv (cd,
                            (ICONV_CONST char **) &inptr, &insize,
                            &outptr, &outsize);
        if (res == (size_t)(-1))
          {
            if (errno == E2BIG)
              ;
            else if (errno == EINVAL)
              break;
            else
              return -1;
          }
# if !defined _LIBICONV_VERSION && !defined __GLIBC__
        else if (res > 0)
          {
            errno = EILSEQ;
            return -1;
          }
# endif
        count += outptr - tmpbuf;
      }
# if defined _LIBICONV_VERSION \
     || !((__GLIBC__ - 0 == 2 && __GLIBC_MINOR__ - 0 <= 1) || defined __sun)
    {
      char *outptr = tmpbuf;
      size_t outsize = tmpbufsize;
      size_t res = iconv (cd, NULL, NULL, &outptr, &outsize);
      if (res == (size_t)(-1))
        return -1;
      count += outptr - tmpbuf;
    }
# endif
    length = count;
# undef tmpbuf
  }
  if (length == 0)
    {
      *lengthp = 0;
      return 0;
    }
  if (*resultp != NULL && *lengthp >= length)
    result = *resultp;
  else
    {
      result = (char *) malloc (length);
      if (result == NULL)
        {
          errno = ENOMEM;
          return -1;
        }
    }
# if defined _LIBICONV_VERSION \
     || !((__GLIBC__ - 0 == 2 && __GLIBC_MINOR__ - 0 <= 1) || defined __sun)
  iconv (cd, NULL, NULL, NULL, NULL);
# endif
  {
    const char *inptr = src;
    size_t insize = srclen;
    char *outptr = result;
    size_t outsize = length;
    while (insize > 0)
      {
        size_t res = iconv (cd,
                            (ICONV_CONST char **) &inptr, &insize,
                            &outptr, &outsize);
        if (res == (size_t)(-1))
          {
            if (errno == EINVAL)
              break;
            else
              goto fail;
          }
# if !defined _LIBICONV_VERSION && !defined __GLIBC__
        else if (res > 0)
          {
            errno = EILSEQ;
            goto fail;
          }
# endif
      }
# if defined _LIBICONV_VERSION \
     || !((__GLIBC__ - 0 == 2 && __GLIBC_MINOR__ - 0 <= 1) || defined __sun)
    {
      size_t res = iconv (cd, NULL, NULL, &outptr, &outsize);
      if (res == (size_t)(-1))
        goto fail;
    }
# endif
    if (outsize != 0)
      abort ();
  }
  *resultp = result;
  *lengthp = length;
  return 0;
 fail:
  {
    if (result != *resultp)
      {
        int saved_errno = errno;
        free (result);
        errno = saved_errno;
      }
    return -1;
  }
# undef tmpbufsize
}
char *
str_cd_iconv (const char *src, iconv_t cd)
{
# if !defined _LIBICONV_VERSION && !defined __GLIBC__
  char *result = NULL;
  size_t length = 0;
  int retval = mem_cd_iconv (src, strlen (src), cd, &result, &length);
  char *final_result;
  if (retval < 0)
    {
      if (result != NULL)
        abort ();
      return NULL;
    }
  final_result =
    (result != NULL ? realloc (result, length + 1) : malloc (length + 1));
  if (final_result == NULL)
    {
      free (result);
      errno = ENOMEM;
      return NULL;
    }
  final_result[length] = '\0';
  return final_result;
# else
  char *result;
  size_t result_size;
  size_t length;
  const char *inptr = src;
  size_t inbytes_remaining = strlen (src);
  result_size = inbytes_remaining;
  {
    size_t approx_sqrt_SIZE_MAX = SIZE_MAX >> (sizeof (size_t) * CHAR_BIT / 2);
    if (result_size <= approx_sqrt_SIZE_MAX / MB_LEN_MAX)
      result_size *= MB_LEN_MAX;
  }
  result_size += 1; 
  result = (char *) malloc (result_size);
  if (result == NULL)
    {
      errno = ENOMEM;
      return NULL;
    }
# if defined _LIBICONV_VERSION \
     || !((__GLIBC__ - 0 == 2 && __GLIBC_MINOR__ - 0 <= 1) || defined __sun)
  iconv (cd, NULL, NULL, NULL, NULL);
# endif
  {
    char *outptr = result;
    size_t outbytes_remaining = result_size - 1;
    for (;;)
      {
        size_t res = iconv (cd,
                            (ICONV_CONST char **) &inptr, &inbytes_remaining,
                            &outptr, &outbytes_remaining);
        if (res == (size_t)(-1))
          {
            if (errno == EINVAL)
              break;
            else if (errno == E2BIG)
              {
                size_t used = outptr - result;
                size_t newsize = result_size * 2;
                char *newresult;
                if (!(newsize > result_size))
                  {
                    errno = ENOMEM;
                    goto failed;
                  }
                newresult = (char *) realloc (result, newsize);
                if (newresult == NULL)
                  {
                    errno = ENOMEM;
                    goto failed;
                  }
                result = newresult;
                result_size = newsize;
                outptr = result + used;
                outbytes_remaining = result_size - 1 - used;
              }
            else
              goto failed;
          }
        else
          break;
      }
# if defined _LIBICONV_VERSION \
     || !((__GLIBC__ - 0 == 2 && __GLIBC_MINOR__ - 0 <= 1) || defined __sun)
    for (;;)
      {
        size_t res = iconv (cd, NULL, NULL, &outptr, &outbytes_remaining);
        if (res == (size_t)(-1))
          {
            if (errno == E2BIG)
              {
                size_t used = outptr - result;
                size_t newsize = result_size * 2;
                char *newresult;
                if (!(newsize > result_size))
                  {
                    errno = ENOMEM;
                    goto failed;
                  }
                newresult = (char *) realloc (result, newsize);
                if (newresult == NULL)
                  {
                    errno = ENOMEM;
                    goto failed;
                  }
                result = newresult;
                result_size = newsize;
                outptr = result + used;
                outbytes_remaining = result_size - 1 - used;
              }
            else
              goto failed;
          }
        else
          break;
      }
# endif
    *outptr++ = '\0';
    length = outptr - result;
  }
  if (length < result_size)
    {
      char *smaller_result = (char *) realloc (result, length);
      if (smaller_result != NULL)
        result = smaller_result;
    }
  return result;
 failed:
  {
    int saved_errno = errno;
    free (result);
    errno = saved_errno;
    return NULL;
  }
# endif
}
#endif
char *
str_iconv (const char *src, const char *from_codeset, const char *to_codeset)
{
  if (*src == '\0' || c_strcasecmp (from_codeset, to_codeset) == 0)
    {
      char *result = strdup (src);
      if (result == NULL)
        errno = ENOMEM;
      return result;
    }
  else
    {
#if HAVE_ICONV
      iconv_t cd;
      char *result;
# if (__GLIBC__ - 0 == 2 && __GLIBC_MINOR__ - 0 <= 1) && !defined _LIBICONV_VERSION
      if (c_strcasecmp (from_codeset, "EUC-KR") == 0
          || c_strcasecmp (to_codeset, "EUC-KR") == 0)
        {
          errno = EINVAL;
          return NULL;
        }
# endif
      cd = iconv_open (to_codeset, from_codeset);
      if (cd == (iconv_t) -1)
        return NULL;
      result = str_cd_iconv (src, cd);
      if (result == NULL)
        {
          int saved_errno = errno;
          iconv_close (cd);
          errno = saved_errno;
        }
      else
        {
          if (iconv_close (cd) < 0)
            {
              int saved_errno = errno;
              free (result);
              errno = saved_errno;
              return NULL;
            }
        }
      return result;
#else
      errno = ENOSYS;
      return NULL;
#endif
    }
}
